﻿//=============================================================================
// TMPlugin - シューティング
// バージョン: 1.2.0
// 最終更新日: 2017/10/23
// 配布元    : http://hikimoki.sakura.ne.jp/
//-----------------------------------------------------------------------------
// Copyright (c) 2016 tomoaky
// Released under the MIT license.
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @plugindesc プレイヤーとイベントに弾を発射する機能を追加します。
 *
 * @author tomoaky (http://hikimoki.sakura.ne.jp/)
 *
 * @param shotKey
 * @type string
 * @desc プレイヤーの弾発射に使用するキー
 * 初期値: A
 * @default A
 *
 * @param holdKey
 * @type string
 * @desc プレイヤーの向き固定に使用するキー
 * 初期値: D
 * @default D
 * 
 * @param holdType
 * @type select
 * @option SWITCH
 * @option HOLD
 * @desc 向き固定方式。SWITCH なら向き固定キーを押すたびに切り替え、
 * HOLD なら押している間だけ切り替え。
 * @default SWITCH
 *
 * @param deadSwitch
 * @type string
 * @desc イベントが戦闘不能になったときにオンになるセルフスイッチ
 * 初期値: A
 * @default A
 *
 * @param clearSwitch
 * @type string
 * @desc イベントを死亡でない非表示状態にするためのセルフスイッチ
 * 初期値: B
 * @default B
 *
 * @param resetDeadSwitch
 * @type boolean
 * @desc マップ移動時に deadSwitch をオフにする
 * 初期値: ON ( false = OFF 無効 / true = ON 有効 )
 * @default true
 *
 * @param leaderShotSe
 * @desc プレイヤー弾発射効果音のファイル名。
 * 初期値: Shot1
 * @default Shot1
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param leaderShotSeParam
 * @type string
 * @desc プレイヤー弾発射効果音のパラメータ。
 * 初期値: {"volume":70, "pitch":150, "pan":0}
 * @default {"volume":70, "pitch":150, "pan":0}
 * 
 * @param defaultDeadAnimeId
 * @desc 戦闘不能時に表示するアニメーション番号の初期値
 * 初期値: 67
 * @default 67
 * @require 1
 * @type animation
 *
 * @param playerDeadEventId
 * @type number
 * @desc 先頭のアクターが戦闘不能時に実行するコモンイベント番号
 * 初期値: 0 ( 0 = 無効 / 1以上 = 該当するコモンイベント起動 )
 * @default 0
 *
 * @param playerDamageEventId
 * @type number
 * @desc 先頭のアクターがダメージを受けた時に実行するコモンイベント番号
 * 初期値: 0 ( 0 = 無効 / 1以上 = 該当するコモンイベント起動 )
 * @default 0
 *
 * @param useGameover
 * @type boolean
 * @desc 全滅時にゲームオーバーシーンへ移行するかどうか
 * 初期値: true ( false = OFF 移行しない / true = ON 移行する )
 * @default true
 *
 * @param maxPlayerBullet
 * @type number
 * @desc 同時に存在できるプレイヤー弾の最大数
 * 初期値: 128
 * @default 128
 *
 * @param maxEnemyBullet
 * @type number
 * @desc 同時に存在できるエネミー弾の最大数
 * 初期値: 128
 * @default 128
 *
 * @param bulletSizeTable
 * @type string
 * @desc 弾の当たり判定の大きさ（ドット数）
 * 初期値: 6,6,6,6,6,6,6,6
 * @default 6,6,6,6,6,6,6,6
 *
 * @param bulletBlendTable
 * @type string
 * @desc 弾のブレンドモード
 * 初期値: 0,0,0,0,0,0,0,0
 * @default 0,0,0,0,0,0,0,0
 *
 * @param equipDummyX
 * @type number
 * @min -9999
 * @desc 装備シーンに表示するダミーのＸ座標
 * 初期値: 408
 * @default 408
 *
 * @param equipDummyY
 * @type number
 * @min -9999
 * @desc 装備シーンに表示するダミーのＹ座標
 * 初期値: 312
 * @default 312
 *
 * @param useLevelUpMessage
 * @type boolean
 * @desc レベルアップメッセージを表示するか
 * 初期値: true ( false = OFF 表示しない / true = ON 表示する )
 * @default true
 *
 * @param padButtons
 * @type string
 * @desc 利用するパッドボタンのコード
 * 初期値: ok,cancel,menu,shift,shot,hold,pageup,pagedown
 * @default ok,cancel,menu,shift,shot,hold,pageup,pagedown
 *
 * @param padButtonNames
 * @type string
 * @desc パッドボタンの名前
 * padButtonsと同じ並び順でボタンの名前を設定してください
 * @default 決定,キャンセル,メニュー,ダッシュ,ショット,向き固定,キャラ変更(前),キャラ変更(次)

 * @param defaultPadButtons
 * @type string
 * @desc パッドボタンの初期配置
 * 初期値: ボタン 1 ～ 12 に対応するコードを設定してください
 * @default cancel,ok,shift,menu,pageup,pagedown,hold,shot,hold,shot,hold,shot
 *
 * @param padConfigCommand
 * @type string
 * @desc パッドボタン配置のコマンド名 (空にすれば機能を無効化)
 * 初期値: パッドボタン配置
 * @default パッドボタン配置
 *
 * @param enemyBashSkill
 * @type number
 * @min 1
 * @desc エネミーの体当たり威力決定スキル
 * 初期値: 1
 * @default 1
 *
 * @param wallHitSkill
 * @type number
 * @min 1
 * @desc 壁接触時の威力決定スキル
 * 初期値: 1
 * @default 1
 *
 * @param viewersPictNum
 * @type number
 * @min 1
 * @desc 閲覧数の1の位を表示するピクチャ番号
 * 初期値: 22
 * @default 22
 *
 * @param shoutPictNum
 * @type number
 * @min 1
 * @desc シャウトを表示するピクチャの番号
 * 初期値: 16
 * @default 16
 *
 * @param oMEventIdVariable
 * @type number
 * @min 1
 * @desc オーバーレイマップイベントが起動された時に、専用のオーバーレイイベントIDを格納するゲーム変数
 * 初期値: 100
 * @default 100
 *
 * @param queenShoutEventIdVariable
 * @type number
 * @min 1
 * @desc 女王叫びの小分類を格納するゲーム変数
 * 初期値: 4
 * @default 4
 *
 * @param queenShoutNormalInterval
 * @type number
 * @min 1
 * @desc 女王の常時叫びの平均インターバルフレーム数
 * 初期値: 360
 * @default 360
 *
 * @param queenShoutAttackInterval
 * @type number
 * @min 1
 * @desc 女王の攻撃時叫びの平均ショット数
 * 初期値: 100
 * @default 100
 *
 * @param queenShoutBeatInterval
 * @type number
 * @min 1
 * @desc 女王の撃破時叫びの平均撃破数
 * 初期値: 10
 * @default 10
 *
 * @param barrierSe
 * @desc バリア時被弾効果音のファイル名。
 * 初期値: h_beam
 * @default h_beam
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param barrierSeParam
 * @type string
 * @desc バリア時被弾効果音のパラメータ。
 * 初期値: {"volume":70, "pitch":100, "pan":0}
 * @default {"volume":70, "pitch":100, "pan":0}
 *
 * @requiredAssets img/system/shootingBullet1
 * @requiredAssets img/system/shootingBullet2
 * @requiredAssets img/system/shootingBullet3
 * @requiredAssets img/system/shootingBullet4
 * @requiredAssets img/system/shootingBullet5
 * @requiredAssets img/system/shootingBullet6
 * @requiredAssets img/system/shootingBullet7
 * @requiredAssets img/system/shootingBullet8
 * 
 * @help
 * TMPlugin - シューティング ver1.2.0
 *
 * 使い方:
 *
 *   このプラグインを動作させるには、プレイヤーやイベントが発射する弾の画像
 *   が必要になります。
 *
 *   弾画像は shootingBullet1.png というファイル名で img/system フォルダに
 *   入れてください。ひとつのファイルには、横に８つ、縦に任意の数の弾画像を
 *   入れることができます。
 *   ファイル名の数字部分を変えて複数のファイルを使用することもできます、
 *   サイズが違う弾、当たり判定が違う弾を作る場合など、必要に応じてファイル
 *   を増やしてください。
 *   shootingBullet8.png まで、最大 8 つのファイルを利用できます。
 *
 *   準備ができたらデータベースで武器のメモ欄に以下のタグを挿入します。
 *   <shotWay:1>
 *   <shotCount:45>
 *   <shotSpeed:0.1>
 *   <shotInterval:15>
 *   <shotType:1>
 *   <shotIndex:8>
 *
 *   この武器をアクターに装備させて、Aキーを押せば弾が発射されます。
 *   弾が当たる敵イベントのメモ欄には <enemy:3> のようなタグを挿入します。
 *   これでこのイベントに 3 番の敵キャラのパラメータが適用され、弾を当てて
 *   HPが 0 になるとセルフスイッチ A がオンになります。
 *
 *   このプラグインは RPGツクールMV Version 1.5.1 で動作確認をしています。
 *
 *   このプラグインはMITライセンスのもとに配布しています、商用利用、
 *   改造、再配布など、自由にお使いいただけます。
 * 
 *
 * プラグインパラメータ補足:
 *
 *   bulletSizeTable
 *     弾画像ファイルごとに当たり判定を設定します、初期設定の 6,6,6,6,6,6,6,6
 *     は shootingBullet1.png ～ shootingBullet8.png までのすべての弾が、弾の
 *     中心から半径６ドットの当たり判定をもつという設定になります。
 *
 *   bulletBlendTable
 *     弾画像の合成方法を設定します、値と合成方法の対応は下記のとおりです。
 *     （ 0 = 通常 / 1 = 加算 / 2 = 乗算 / 3 = スクリーン ）
 *     bulletSizeTable と同様に弾画像ファイルの数だけ設定する必要があります。
 *
 *   useLevelUpMessage
 *     弾による敵イベントの撃破で経験値を獲得した際に、アクターのレベルアップ
 *     をメッセージウィンドウで表示するかどうかを設定します。
 *     アクション要素が強いゲームでは非表示にすることをおすすめします。
 *
 *
 * プラグインコマンド:
 *
 *   startAutoShot
 *     このコマンドが実行されるとプレイヤーキャラクターが自動的に弾を撃つよう
 *     になります。この変更はパーティメンバーにも適用されます。
 * 
 *   stopAutoShot
 *     このコマンドが実行されるとプレイヤーキャラクターの自動射撃が止まります、
 *     この変更はパーティメンバーにも適用されます。
 *
 *   equipSpecialWeapon 0 1 3
 *     イベントに特殊弾を装備させます。コマンド名に続く数値は左から
 *     対象イベントID（0で自身）、特殊弾ID（0で消去）、以降弾ごとのパラメータ
 *     となります。特殊弾についてはメモ欄タグの<specialWeapon:1,1,1,...>参照。
 *
 *   nwayShot 3 0.4 0 0.1 60 1 3 1
 *     このコマンドを実行したイベントが弾を発射します、コマンド名に続く数値は
 *     左から 弾数、間隔、角度、速度、寿命、タイプ、インデックス、スキル番号
 *     となります。
 *     タイプが 1 で、インデックスが 3 なら、弾画像として shootingBullet1.png
 *     の最上段、左から４つ目を使用します。（インデックスは 0 が先頭です）
 *
 *   nwayAim 3 0.4 0 0.1 60 1 3 1
 *     このコマンドを実行したイベントが自機狙いの弾を発射します。
 *     角度が 0 以外の場合は、自機がいる方向にその値を加算した角度で発射され
 *     ます。
 *
 *   nallShot 8 0 0.1 60 1 3 1
 *     このコマンドを実行したイベントが全方位に弾を発射します、コマンド名に
 *     続く数値は左から 弾数、角度、速度、寿命、タイプ、インデックス、
 *     スキル番号 となります。
 *     弾の間隔は全方位に発射されるように自動で調整されます。
 *
 *   nallAim 8 0 0.1 60 1 3 1
 *     nallShotの自機狙い版です。
 *
 *   stopPlayerShot
 *     プレイヤー（パーティメンバー含む）の弾発射を手動、自動問わず禁止します。
 *
 *   startPlayerShot
 *     プレイヤー（パーティメンバー含む）の弾発射禁止状態を解除します。
 * 
 *   stopPlayerShot message
 *     イベントコマンド『文章の表示』実行中のみプレイヤー（パーティメンバー含む）
 *     の発射弾を手動、自動問わず禁止します。
 * 
 *   startPlayerShot message
 *     stopPlayerShot message の効果を解除します。
 *
 *   stopEnemyShot
 *     イベントの弾発射を禁止します、並列イベントで弾を発射している場合は
 *     弾発射のコマンドのみが無効化され、そのほかのコマンドは実行されます。
 *
 *   startEnemyShot
 *     イベントの弾発射禁止状態を解除します。
 *
 *   stopEnemyShot message
 *     イベントコマンド『文章の表示』実行中のみイベントの弾発射を禁止します。
 * 
 *   startEnemyShot message
 *     stopEnemyShot message の効果を解除します。
 *
 *   deletePlayerBullets
 *     プレイヤー（パーティメンバー含む）が発射したすべての弾を消去します。
 *
 *   deleteEnemyBullets
 *     イベントが発射したすべての弾を消去します。
 *
 *   forceShot 0
 *     プレイヤーのショット操作を強制実行します。このコマンドはディレイを
 *     無視して弾を発射します。数値はパーティの先頭を 0 とした並び順です、
 *     0 が指定されていれば先頭のキャラクターのみが弾を発射します。
 *     数値を省略した場合はパーティ全員が弾を発射します。
 *
 *   pushShout 攻撃
 *     女王のシャウトを追加予約します。
 *     パラメータには「開始」「攻撃」「被弾」「常時」「初回」「撃破」のいずれかを指定します。
 *
 *   stopShout
 *     今叫んでいるシャウト終了後に、女王のシャウトを停止します。
 *
 *   terminateShout
 *     女王のシャウトを強制停止します。
 *
 *   interruptShout 攻撃
 *     女王のシャウトを強制停止し、新たにシャウトを追加します。
 *     パラメータには「開始」「攻撃」「被弾」「常時」「初回」「撃破」のいずれかを指定します。
 *
 *   resetShoutTimes
 *     女王のシャウト消費回数を初期化します。
 *
 *
 * メモ欄タグ（アクター、装備、ステート）:
 *
 *   <shotWay:3>
 *     一度に発射される弾の数を設定します。
 *
 *   <shotSpace:0.4>
 *     一度に発射される弾同士の間隔（角度）を設定します。
 *
 *   <shotSpeed:0.1>
 *     弾の移動速度を設定します。
 *
 *   <shotCount:60>
 *     弾が消えるまでの時間をフレーム数で設定します。
 *
 *   <shotInterval:20>
 *     再発射までの発射不可時間をフレーム数で設定します。
 *
 *   <shotBurst:3>
 *     追加インターバルを設定するまでの発射数。
 *
 *   <shotBurstInterval:14>
 *     shotBurstで設定した弾数を発射した場合、
 *     再発射までの発射不可時間にこの値が加算されます。
 * 
 *   <shotInvincible:60>
 *     被弾により発生する無敵時間をフレーム数で設定します。
 *
 *   <specialWeapon:1,1,1,...>
 *     発射する弾を特殊なものに変換します。アクター＜武器＜ステートの順に優先されます。
 *     引数は1番目が弾のタイプ、2番目以降は弾ごとに指定すべきパラメータを入れます。
 *     type.1 画面端バウンド弾  param1:バウンド回数
 *     type.2 追尾弾  param1:追尾対象[0プレイヤー,1最も近い敵]  param2:追尾強度[1～90程度]
 *     type.3 落下弾  param1:重力強度
 *     type.4 落下地這い弾  param1:重力強度
 *
 * メモ欄タグ（装備、ステート）:
 *
 *   <shotIntervalRate:1.4>
 *     再発射までの発射不可時間を倍率で設定します。
 *     shotInterval による加算補正よりも後に計算されます。
 *
 * メモ欄タグ（武器、ステート）:
 *
 *   <shotType:1>
 *     弾のグラフィックとして使う画像ファイルを設定します。値が 1 なら
 *     shootingBullet1.png を使用します。
 *     武器とステートの両方にこのタグがある場合、ステートのものを優先します。
 *
 *   <shotIndex:3>
 *     shotTypeタグで選択した画像ファイルの何番目の弾を使用するか設定します。
 *     武器とステートの両方にこのタグがある場合、ステートのものを優先します。
 *
 *   <shotSkill:1>
 *     弾が相手に当たったときのダメージ計算に使うスキルを設定します。
 *     武器とステートの両方にこのタグがある場合、ステートのものを優先します。
 *
 *
 * メモ欄タグ（スキル）:
 *
 *   <mapThrough>
 *     マップの通行不可タイルと接触しても弾が消えなくなります。
 *
 *   <penetrate>
 *     キャラクターと接触しても弾は消えずに貫通します。キャラクターが同じ弾に
 *     複数回ダメージを受けることはありません。
 *
 *   <bulletAnime:1>
 *     弾がキャラクターにヒットした際に、指定した番号のアニメーションを
 *     被弾したキャラクターに表示します。
 *
 *
 * メモ欄タグ（イベント）:
 *
 *   <enemy:1>
 *     イベントのパラメータとして利用する敵キャラ番号を設定します。
 *
 *   <cw:0.375>
 *     イベントと弾の当たり判定サイズ（横幅）をイベントの中心から左（右）端
 *     までの長さで設定します。値は 1.0 でマップのタイル１マス分になります。
 *     このタグがない場合は初期値として 0.375 を使用します。
 *
 *   <ch:0.75>
 *     イベントと弾の当たり判定サイズ（高さ）をイベントの足元から上端までの
 *     長さで設定します。値は 1.0 でマップのタイル１マス分になります。
 *     このタグがない場合は初期値として 0.75 を使用します。
 *
 *   <shiftcx:0>
 *     イベントと弾の当たり判定のX座標をデフォルトの位置から右にずらします。
 *     値は 1.0 でマップのタイル１マス分になります。（マイナスで左方向）
 *     このタグがない場合は初期値として 0 を使用します。
 *
 *   <shiftcy:0>
 *     イベントと弾の当たり判定のY座標をデフォルトの位置から下にずらします。
 *     値は 1.0 でマップのタイル１マス分になります。（マイナスで上方向）
 *     このタグがない場合は初期値として 0 を使用します。
 *
 *   <shiftFiringY:0>
 *     イベントの弾発射位置（Ｙ座標）をずらします。
 *     値は 1.0 でマップのタイル１マス分になります。
 *     値が正なら下、負なら上方向へずらします。
 *     通常は当たり判定の矩形の中心から弾が発射されますが、不都合がある場合は
 *     このタグで調整してください。
 *
 *   <shiftFiringX:0>
 *     イベントの弾発射位置（X座標）をずらします。
 *     値は 1.0 でマップのタイル１マス分になります。
 *     値が正なら右、負なら左方向へずらします。
 *     通常は当たり判定の矩形の中心から弾が発射されますが、不都合がある場合は
 *     このタグで調整してください。
 *
 *
 * メモ欄タグ（アクター）:
 *
 *   <cw:0.375>
 *     イベント用のものと同じです。
 *
 *   <ch:0.75>
 *     イベント用のものと同じです。
 *
 *   <shiftcx:0>
 *     イベント用のものと同じです。
 *
 *   <shiftcy:0>
 *     イベント用のものと同じです。
 *
 *   <shiftFiringY:0>
 *     イベント用のものと同じです。
 *
 *   <shiftFiringX:0>
 *     イベント用のものと同じです。
 *
 *   <deadCharacter:!Flame,5>
 *     このアクターが戦闘不能になったとき、歩行グラフィックを変更します。
 *     この例では !Flame.png の下段、左から２番目のグラフィックが採用されます。
 *
 *
 * メモ欄タグ（アクター、敵キャラ）:
 *
 *   <deadAnime:67>
 *     戦闘不能ステートが付加されたときに表示するアニメーションを設定します。
 *
 *
 * 併用可能（動作確認済み）プラグイン:
 *
 *   SAN_AnalogMove.js ver1.40
 *   作者: サンシロさん（http://rev2nym.blog.fc2.com/）
 *
 *   CharacterPopupDamage.js Version 1.5.0
 *   作者: トリアコンタンさん（http://triacontane.blogspot.jp/）
 *
 *   CommonPopupCore.js ver1.05
 *   GetInformation.js ver1.15
 *   作者: Yanaさん（https://twitter.com/yanatsuki_）
 *
 *   上記プラグインを併用したことによる不具合の報告は、併用プラグインの
 *   作者様ではなく、必ずtomoakyへお願いします。
 *
 *
 * その他注意点など:
 *
 *   shotWay や shotSpace などのメモ欄タグは、アクター、装備、ステートの
 *   合計値が採用されます。shotWay が 0 の場合、または shotCount が 0 の
 *   場合は弾が発射されないか、発射後すぐに消滅してしまいます、素手の状態
 *   でも弾を撃ちたい場合はアクターにも shotWay と shotCount タグを設定する
 *   必要があります。
 *
 *   nwayShot などのコマンドに味方対象のスキルを設定した場合、イベント同士
 *   での攻撃が表現できますが、スキルの設定によっては命中判定が正しく機能
 *   しません。
 *   弾が当たらない場合はスキルの使用効果にステート付加 0 %などの無意味な
 *   効果を設定することでこの問題を回避することができます。
 *
 *   パッドボタン配置は save フォルダ内の config.rpgsave に保存されます、
 *   このファイルが削除されるまでは初期配置の設定を変更しても適用されません。
 */

var Imported = Imported || {};
Imported.TMShooting = true;

var TMPlugin = TMPlugin || {};
TMPlugin.Shooting = {};
TMPlugin.Shooting.Parameters = PluginManager.parameters('TMShooting');
TMPlugin.Shooting.HoldType           =   TMPlugin.Shooting.Parameters['holdType'] || 'SWITCH';
TMPlugin.Shooting.DeadSwitch         =   TMPlugin.Shooting.Parameters['deadSwitch'] || 'A';
TMPlugin.Shooting.ClearSwitch         =   TMPlugin.Shooting.Parameters['clearSwitch'] || 'B';
TMPlugin.Shooting.ResetDeadSwitch    =   TMPlugin.Shooting.Parameters['resetDeadSwitch'] === '1';
TMPlugin.Shooting.DefaultDeadAnimeId = +(TMPlugin.Shooting.Parameters['defaultDeadAnimeId'] || 0);
TMPlugin.Shooting.LeaderShotSe       = JSON.parse(TMPlugin.Shooting.Parameters['leaderShotSeParam'] || '{}');
TMPlugin.Shooting.LeaderShotSe.name  =   TMPlugin.Shooting.Parameters['leaderShotSe'] || '';
TMPlugin.Shooting.PlayerDeadEventId  = +(TMPlugin.Shooting.Parameters['playerDeadEventId'] || 0);
TMPlugin.Shooting.UseGameover        = JSON.parse(TMPlugin.Shooting.Parameters['useGameover']);
TMPlugin.Shooting.MaxPlayerBullet    = +(TMPlugin.Shooting.Parameters['maxPlayerBullet'] || 128 );
TMPlugin.Shooting.MaxEnemyBullet     = +(TMPlugin.Shooting.Parameters['maxEnemyBullet'] || 128 );
TMPlugin.Shooting.EquipDummyX        = +(TMPlugin.Shooting.Parameters['equipDummyX'] || 408);
TMPlugin.Shooting.EquipDummyY        = +(TMPlugin.Shooting.Parameters['equipDummyY'] || 312);
TMPlugin.Shooting.UseLevelUpMessage  = JSON.parse(TMPlugin.Shooting.Parameters['useLevelUpMessage']);
TMPlugin.Shooting.PadButtons         =   TMPlugin.Shooting.Parameters['padButtons'].split(',');
TMPlugin.Shooting.PadButtonNames     =   TMPlugin.Shooting.Parameters['padButtonNames'].split(',');
TMPlugin.Shooting.DefaultPadButtons  =   TMPlugin.Shooting.Parameters['defaultPadButtons'].split(',');
TMPlugin.Shooting.PadConfigCommand   =   TMPlugin.Shooting.Parameters['padConfigCommand'];

TMPlugin.Shooting.EnemyBashSkillId   =   TMPlugin.Shooting.Parameters['enemyBashSkill'];
TMPlugin.Shooting.WallHitSkillId   =   TMPlugin.Shooting.Parameters['wallHitSkill'];
TMPlugin.Shooting.PlayerDamageEventId  = +(TMPlugin.Shooting.Parameters['playerDamageEventId'] || 0);
TMPlugin.Shooting.OMEventIDVariable  = TMPlugin.Shooting.Parameters['oMEventIdVariable'];
TMPlugin.Shooting.ViewersPictNum     = +(TMPlugin.Shooting.Parameters['viewersPictNum'] || 22);
TMPlugin.Shooting.ShoutPictNum       = +(TMPlugin.Shooting.Parameters['shoutPictNum'] || 16);
TMPlugin.Shooting.QueenShoutEventIdVariable  = +(TMPlugin.Shooting.Parameters['queenShoutEventIdVariable'] || 4);
TMPlugin.Shooting.QueenShoutNormalInterval = +(TMPlugin.Shooting.Parameters['queenShoutNormalInterval'] || 360);
TMPlugin.Shooting.QueenShoutAttackInterval = +(TMPlugin.Shooting.Parameters['queenShoutAttackInterval'] || 100);
TMPlugin.Shooting.QueenShoutBeatInterval = +(TMPlugin.Shooting.Parameters['queenShoutBeatInterval'] || 10);

TMPlugin.Shooting.BarrierSe = JSON.parse(TMPlugin.Shooting.Parameters['barrierSeParam'] || '{}');
TMPlugin.Shooting.BarrierSe.name = TMPlugin.Shooting.Parameters['barrierSe'] || '';


TMPlugin.Shooting.BulletSizeTable = [0];
TMPlugin.Shooting.Parameters['bulletSizeTable'].split(',').forEach(function(size) {
  TMPlugin.Shooting.BulletSizeTable.push(+size / 48);
});

TMPlugin.Shooting.BulletBlendTable = [0];
TMPlugin.Shooting.Parameters['bulletBlendTable'].split(',').forEach(function(blendMode) {
  TMPlugin.Shooting.BulletBlendTable.push(+blendMode);
});


TMPlugin.Shooting.SmoothMode = false;

function Game_Bullet() {
  this.initialize.apply(this, arguments);
}

function Game_PlayerBullet() {
    this.initialize.apply(this, arguments);
}

function Game_EnemyBullet() {
    this.initialize.apply(this, arguments);
}

(function() {

  //-----------------------------------------------------------------------------
  // Input
  //

  if (TMPlugin.Shooting.Parameters['shotKey']) {
    Input.keyMapper[TMPlugin.Shooting.Parameters['shotKey'].charCodeAt()] = 'shot';
  }

  if (TMPlugin.Shooting.Parameters['holdKey']) {
    Input.keyMapper[TMPlugin.Shooting.Parameters['holdKey'].charCodeAt()] = 'hold';
  }

  //-----------------------------------------------------------------------------
  // ConfigManager
  //

  ConfigManager.getPadButton = function(id) {
    return Input.gamepadMapper[id];
  };
  
  ConfigManager.setPadButton = function(id, code) {
    Input.gamepadMapper[id] = code;
  };

  var _ConfigManager_makeData = ConfigManager.makeData;
  ConfigManager.makeData = function() {
    var config = _ConfigManager_makeData.call(this);
    for (var i = 0; i < 12; i++) {
      var key = 'padButton' + i;
      config[key] = this.getPadButton(i);
    }
    return config;
  };

  var _ConfigManager_applyData = ConfigManager.applyData;
  ConfigManager.applyData = function(config) {
    _ConfigManager_applyData.call(this, config);
    for (var i = 0; i < 12; i++) {
      var key = 'padButton' + i;
      this.setPadButton(i, this.readPadButton(config, i));
    }
  };

  ConfigManager.readPadButton = function(config, id) {
    var key = 'padButton' + id;
    return config[key] || TMPlugin.Shooting.DefaultPadButtons[id];
  };

  //-----------------------------------------------------------------------------
  // Game_System
  //

  var _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function() {
    _Game_System_initialize.call(this);
    this._playerShotEnabled = true;
    this._playerShotEnabledMessage = true;
    this._enemyShotEnabled = true;
    this._enemyShotEnabledMessage = true;
  };

  Game_System.prototype.isPlayerShotEnabled = function() {
    if (!this._playerShotEnabledMessage && $gameMessage.isBusy()) return false;
    return this._playerShotEnabled;
  };

  Game_System.prototype.setPlayerShotEnabled = function(value, flag) {
    if (flag === 'message') {
      this._playerShotEnabledMessage = value;
    } else {
      this._playerShotEnabled = value;
    }
  };

  Game_System.prototype.isEnemyShotEnabled = function() {
    if (!this._enemyShotEnabledMessage && $gameMessage.isBusy()) return false;
    return this._enemyShotEnabled;
  };

  Game_System.prototype.setEnemyShotEnabled = function(value, flag) {
    if (flag === 'message') {
      this._enemyShotEnabledMessage = value;
    } else {
      this._enemyShotEnabled = value;
    }
  };
  
  //-----------------------------------------------------------------------------
  // Game_Action
  //

  var _Game_Action_setSubject = Game_Action.prototype.setSubject;
  Game_Action.prototype.setSubject = function(subject) {
    _Game_Action_setSubject.call(this, subject);
    if (this._subjectActorId === 0 && this._subjectEnemyIndex < 0) {
      this._subjectEnemyIndex = subject.screenX();
    }
  };

  var _Game_Action_subject = Game_Action.prototype.subject;
  Game_Action.prototype.subject = function() {
    if (this._subjectActorId === 0 && this._subjectEnemyIndex < 0) {
      return $gameMap.event(-this._subjectEnemyIndex).battler();
    } else {
      return _Game_Action_subject.call(this);
    }
  };

  //-----------------------------------------------------------------------------
  // Game_Actor
  //

  var _Game_Actor_shouldDisplayLevelUp = Game_Actor.prototype.shouldDisplayLevelUp;
  Game_Actor.prototype.shouldDisplayLevelUp = function() {
    if (!TMPlugin.Shooting.UseLevelUpMessage) return false;
    return _Game_Actor_shouldDisplayLevelUp.call(this);
  };

  var _Game_Actor_refresh = Game_Actor.prototype.refresh;
  Game_Actor.prototype.refresh = function() {
    _Game_Actor_refresh.call(this);
    this.refreshShotParam();
    this.refreshDeadCharacter();
  };

  Game_Actor.prototype.refreshDeadCharacter = function() {
    var actor = this.actor();
    var deadCharacter = actor.meta.deadCharacter;
    if (deadCharacter) {
      deadCharacter = deadCharacter.split(',');
      this._deadCharacterName = deadCharacter[0];
      this._deadCharacterIndex = +deadCharacter[1];
    } else {
      this._deadCharacterName = null;
      this._deadCharacterIndex = null;
    }
  };
  
  Game_Actor.prototype.refreshShotParam = function() {
    this._shotParams = {};
    var data = this.actor();
    this._shotParams.way      = +(data.meta.shotWay || 0);
    this._shotParams.space    = +(data.meta.shotSpace || 0);
    this._shotParams.speed    = +(data.meta.shotSpeed || 0);
    this._shotParams.count    = +(data.meta.shotCount || 0);
    this._shotParams.interval = +(data.meta.shotInterval || 0);
    this._shotParams.burstInterval = +(data.meta.shotBurstInterval || 0);
    this._shotParams.burst = +(data.meta.shotBurst || 0);
    this._shotParams.invincible = +(data.meta.shotInvincible || 0);
    this._shotParams.special = (data.meta.specialWeapon || null);
    this._shotParams.skillId = +(data.meta.shotSkill || this.attackSkillId());
    this._shotParams.type    = +(data.meta.shotType || 0);
    this._shotParams.index   = +(data.meta.shotIndex || 0);
    var items = this.equips().concat(this.states());
    for (var i = 0; i < items.length; i++) {
      var item = items[i];
      if (item) {
        this._shotParams.way      += +(item.meta.shotWay || 0);
        this._shotParams.space    += +(item.meta.shotSpace || 0);
        this._shotParams.speed    += +(item.meta.shotSpeed || 0);
        this._shotParams.count    += +(item.meta.shotCount || 0);
        this._shotParams.interval += +(item.meta.shotInterval || 0);
        this._shotParams.burstInterval += +(item.meta.shotBurstInterval || 0);
        this._shotParams.burst += +(item.meta.shotBurst || 0);
        this._shotParams.invincible += +(item.meta.shotInvincible || 0);
      }
    }
    for (var i = 0; i < items.length; i++) {
      var item = items[i];
      if (item) {
        this._shotParams.interval *= +(item.meta.shotIntervalRate || 1);
      }
    }
    var weapon = this.weapons()[0];
    if (weapon) {
      if (weapon.meta.shotType)  this._shotParams.type    += +weapon.meta.shotType;
      if (weapon.meta.shotType)  this._shotParams.index   += +weapon.meta.shotIndex;
      if (weapon.meta.shotSkill) this._shotParams.skillId = +(weapon.meta.shotSkill);
      if (weapon.meta.specialWeapon) this._shotParams.special = weapon.meta.specialWeapon;
    } else {
//      this._shotParams.type    = 1;
//      this._shotParams.index   = 0;
//      this._shotParams.skillId = this.attackSkillId();
    }
    var items = this.states();
    for (var i = 0; i < items.length; i++) {
      var item = items[i];
      if (item) {
        if (item.meta.shotType)  this._shotParams.type = +item.meta.shotType;
        if (item.meta.shotIndex) this._shotParams.index = +item.meta.shotIndex;
        if (item.meta.shotSkill) this._shotParams.skillId = +item.meta.shotSkill;
        if (item.meta.specialWeapon) this._shotParams.special = item.meta.specialWeapon;
      }
    }
  };
  
  Game_Actor.prototype.shotParams = function() {
    return this._shotParams;
  };
  
  Game_Actor.prototype.deadCharacterName = function() {
    return this._deadCharacterName;
  };

  Game_Actor.prototype.deadCharacterIndex = function() {
    return this._deadCharacterIndex;
  };

  // 特殊弾を返す
  Game_Actor.prototype.specialWeapon = function() {
    // 後でここにexShotのspecialWeapon切り替えも突っ込む（雑に）
    if (!this._shotParams.special) return null;
    var special = {};
    var args = this._shotParams.special.split(',').map(Number);
    special.type = args[0];
    for ( var i = 1;i<args.length;i++ ){
      special['param'+i] = args[i];
    }
    return special;
  };

  //-----------------------------------------------------------------------------
  // Game_Battler
  //

  // 特殊弾を返す
  Game_Battler.prototype.specialWeapon = function() {
    var special = null;
    return special;
  };

  //-----------------------------------------------------------------------------
  // Game_Enemy
  //

  // 特殊弾を装備
  Game_Enemy.prototype.equipSpecialWeapon = function(args) {
    if (args[0]==0){
      this._specialWeapon = null;
      return;
    }
    this._specialWeapon = {};
    this._specialWeapon.type = args[0];
    for (var i = 1;i<args.length;i++){
      this._specialWeapon['param'+i] = args[i];
    }
  };

  // 特殊弾を返す
  Game_Enemy.prototype.specialWeapon = function() {
    if (!this._specialWeapon) return null;
    return this._specialWeapon;
  };
  
  //-----------------------------------------------------------------------------
  // Game_Map
  //

  // セットアップ
  var _Game_Map_setup = Game_Map.prototype.setup;
  Game_Map.prototype.setup = function(mapId) {
    _Game_Map_setup.call(this, mapId);
    this.setupBullets();
    this.createBulletPassageTable();
    if (TMPlugin.Shooting.ResetDeadSwitch) {
      this.events().forEach(function(event) {
        var key = [mapId, event.eventId(), TMPlugin.Shooting.DeadSwitch];
        $gameSelfSwitches.setValue(key, false);
      });
    }
  };

  // 弾のセットアップ
  Game_Map.prototype.setupBullets = function() {
    this._playerBullets = [];
    this._alivePlayerBullets = [];
    this._blankPlayerBullets = [];
    for (var i = 0; i < TMPlugin.Shooting.MaxPlayerBullet; i++) {
      this._playerBullets.push(new Game_PlayerBullet());
      this._blankPlayerBullets.push(i);
    }
    this._enemyBullets = [];
    this._aliveEnemyBullets = [];
    this._blankEnemyBullets = [];
    for (var i = 0; i < TMPlugin.Shooting.MaxEnemyBullet; i++) {
      this._enemyBullets.push(new Game_EnemyBullet());
      this._blankEnemyBullets.push(i);
    }
  };

  Game_Map.prototype.playerBullets = function() {
    return this._playerBullets;
  };

  Game_Map.prototype.enemyBullets = function() {
    return this._enemyBullets;
  };
  
  // 弾の通行チェックに利用するテーブルを作成する
  Game_Map.prototype.createBulletPassageTable = function() {
    this._bulletPassageTable = [];
    var flags = this.tilesetFlags();
    for (var x = 0; x < $dataMap.width; x++) {
      this._bulletPassageTable.push([]);
      for (var y = 0; y < $dataMap.height; y++) {
        var tiles = this.layeredTiles(x, y);
        var passage = false;
        for (var i = 0; i < tiles.length; i++) {
          var flag = flags[tiles[i]];
          if ((flag & 0x10) !== 0) continue;
          if ((flag & 0x0f) !== 0x0f) {
            passage = true;
            break;
          }
          if ((flag & 0x0f) !== 0) break;
        }
        this._bulletPassageTable[x][y] = passage;
      }
    }
  };

  // 弾の通行チェック
  Game_Map.prototype.checkPassageBullet = function(x, y) {
    return this.isValid(x, y) && this._bulletPassageTable[x][y];
  };

  // フレーム更新
  var _Game_Map_update = Game_Map.prototype.update;
  Game_Map.prototype.update = function(sceneActive) {
    _Game_Map_update.call(this, sceneActive);
    this.updateBullets();
    this.updatePlayerCollideWithEvents();
  };

  // 弾の更新
  Game_Map.prototype.updateBullets = function() {
    for (var i = this._alivePlayerBullets.length - 1; i >= 0; i--) {
      var index = this._alivePlayerBullets[i];
      if (!this._playerBullets[index].update()) {
        this._alivePlayerBullets.splice(i, 1);
        this._blankPlayerBullets.push(index);
      }
    }
    for (var i = this._aliveEnemyBullets.length - 1; i >= 0; i--) {
      var index = this._aliveEnemyBullets[i];
      if (!this._enemyBullets[index].update()) {
        this._aliveEnemyBullets.splice(i, 1);
        this._blankEnemyBullets.push(index);
      }
    }
  };

  // 弾の追加
  Game_Map.prototype.addBullet = function(x, y, z, vx, vy, angle, count, type,
                                          index, skillId, ownerId, special = null) {
    if (ownerId < 0) {
      if (this._blankPlayerBullets.length > 0) {
        var bulletIndex = this._blankPlayerBullets.shift();
        this._playerBullets[bulletIndex].setup(x, y, z, vx, vy, angle, count,
                                               type, index, skillId, ownerId, special);
        this._alivePlayerBullets.push(bulletIndex);
      }
    } else {
      if (this._blankEnemyBullets.length > 0) {
        var bulletIndex = this._blankEnemyBullets.shift();
        this._enemyBullets[bulletIndex].setup(x, y, z, vx, vy, angle, count,
                                              type, index, skillId, ownerId, special);
        this._aliveEnemyBullets.push(bulletIndex);
      }
    }
  };

  // プレイヤー弾の全削除
  Game_Map.prototype.clearPlayerBullets = function() {
    this._alivePlayerBullets.forEach(function(index) {
      this._playerBullets[index].erase();
    }, this);
    this._blankPlayerBullets.concat(this._alivePlayerBullets);
    this._alivePlayerBullets = [];
  };

  // エネミー弾の全削除
  Game_Map.prototype.clearEnemyBullets = function() {
    this._aliveEnemyBullets.forEach(function(index) {
      this._enemyBullets[index].erase();
    }, this);
    this._blankEnemyBullets.concat(this._aliveEnemyBullets);
    this._aliveEnemyBullets = [];
  };

  // すべての弾を削除
  Game_Map.prototype.clearAllBullets = function() {
    this.clearPlayerBullets();
    this.clearEnemyBullets();
  };

  // プレイヤーのイベントとの接触判定とその処理
  Game_Map.prototype.updatePlayerCollideWithEvents = function() {
    if ($gameMap.isEventRunning()) return; // イベント起動時は無視
    var events = $gamePlayer.collide_events();
    events.forEach( function(event){
      // イベント起動（接触イベントとして設定されているもののみ）
      if (!$gameMap.isEventRunning()) {
        if (event.isTriggerIn([1,2])) {
          event.start();
        }
      }
      if ($gamePlayer.isShotInvincible() || $gamePlayer._suyapiShotMode) return; // 無敵時はダメージ系は無視
      // ダメージ（プレイヤーと同じプライオリティに設定された、エネミー情報を持つイベントのみ）
      if (event.isNormalPriority() === true && event._enemyId != 0) {
        if($gamePlayer.hasBarrier()){ // バリアあり
          $gamePlayer.reduceBarrier();
          var dummyBullet = new Game_PlayerBullet();
          dummyBullet.setup(0, 0, 0, 0, 0, 0, 0, 0, 0, 1, $gamePlayer.ownerId());
          dummyBullet.executeDamage(event);
        }else{
          var dummyBullet = new Game_EnemyBullet();
          dummyBullet.setup(0, 0, 0, 0, 0, 0, 0, 0, 0, TMPlugin.Shooting.EnemyBashSkillId, event.ownerId());
          dummyBullet.executeDamage($gamePlayer);
        }
      }
    });
  };

  //-----------------------------------------------------------------------------
  // Collide_Rect
  //

  function Collide_Rect() {
    this.initialize.apply(this, arguments);
  }

  // 初期化
  Collide_Rect.prototype.initialize = function(l,t,w,h) {
    this._left = l;
    this._top = t;
    this._width = w;
    this._height = h;
  };

  Collide_Rect.prototype.centerX = function(){
    return this._left + this._width / 2;
  };
  Collide_Rect.prototype.centerY = function(){
    return this._top + this._height / 2;
  };

  // 2つのRect間での接触判定
  Collide_Rect.prototype.isCollidedWith = function(cr) {
    return this._left <= cr._left + cr._width &&
           cr._left <= this._left + this._width &&
           this._top <= cr._top + cr._height &&
           cr._top <= this._top + this._height
  };


  //-----------------------------------------------------------------------------
  // Game_Bullet
  //

  // 初期化
  Game_Bullet.prototype.initialize = function() {
    this._opacity = 0;
  };

  // セットアップ
  Game_Bullet.prototype.setup = function(x, y, z, vx, vy, angle, count, type,
                                         index, skillId, ownerId, special = null) {
    this._opacity = 255;
    this._x = x;
    this._y = y;
    this._z = z;
    this._vx = vx;
    this._vy = vy;
    this._angle = angle;
    this._count = count;
    if (this._type !== type) {
      this._type = type;
      this._bulletName = 'shootingBullet' + type;
      this._collideSize = TMPlugin.Shooting.BulletSizeTable[type];
    }
    this._bulletIndex = index;
    this._skillId = skillId;
    this._ownerId = ownerId;
    var skill = $dataSkills[skillId];
    this._mapCollide = !skill.meta.mapThrough;
    this._penetrate = skill.meta.penetrate;
    this._friendly = skill.scope >= 7;
    this._collidedCharacters = [ownerId];
    this._specialWeapon = null;
    if (!special) return;
    this._specialWeapon = special;
    // 特殊弾パラメータ初期化
    if (special.type == 1){ // バウンド弾
      this._swParam = this._specialWeapon.param1;
    }else if(special.type == 2){ // 追尾弾
      var target = this._specialWeapon.param1;
      this._trackingTarget = $gamePlayer;
      this.searchTarget();
      this._trackingPower = this._specialWeapon.param2;
      this._trachingFrame = (this._specialWeapon.param3 || 9999); // 設定ナシの場合ひとまず9999フレーム
    }else if(special.type == 3){ // 落下弾
      this._swParam = this._specialWeapon.param1;
    }else if(special.type == 4){ // 落下地這い弾
      this._swParam = this._specialWeapon.param1;
      this._swParam2 = vx; // 速度を記録
    }
  };

  // 追尾弾の対象探索
  Game_Bullet.prototype.searchTarget = function() {
    if (this._specialWeapon.param1 != 0) {
      var events = $gameMap.events();
      var arrayEvents = Object.keys(events).map(function (key) {return events[key]});
      var bullet = this;
      arrayEvents = arrayEvents.filter(event => bullet.canTrack(event));
      arrayEvents.sort(function(a,b){return bullet.dfcs(a)-bullet.dfcs(b);});
      this._trackingTarget = arrayEvents[0];
    }
  };

  // 追尾弾が対象を追尾できるかどうか判定
  Game_Bullet.prototype.canTrack = function(character) {
    return character.battler() && character.battler().isAlive() && !character._through;
  };

  // 追尾弾での対象探索に使用するソート関数（簡易距離計算）
  Game_Bullet.prototype.dfcs = function(character) {
    var rect = character.collideRect();
    return (this._x - rect.centerX())**2 + (this._y - rect.centerY())**2;
  };

  // 存在状態判定
  Game_Bullet.prototype.isExist = function() {
    return this._count > 0;
  };

  // 不透明度の取得
  Game_Bullet.prototype.opacity = function() {
    return this._opacity;
  };

  // 画像ファイル名の取得
  Game_Bullet.prototype.bulletName = function() {
    return this._bulletName;
  };

  // 画像ファイルインデックスの取得
  Game_Bullet.prototype.bulletIndex = function() {
    return this._bulletIndex;
  };

  // 弾タイプの取得
  Game_Bullet.prototype.type = function() {
    return this._type;
  };

  // 角度の取得
  Game_Bullet.prototype.angle = function() {
    if (this._specialWeapon && this._specialWeapon.type == 2){ // 追尾弾
      return Math.atan2(this._vy, this._vx);
    }
    return this._angle;
  };

  // 画面 X 座標の取得
  Game_Bullet.prototype.screenX = function() {
    var tw = $gameMap.tileWidth();
    return Math.round($gameMap.adjustX(this._x) * tw);
  };

  // 画面 Y 座標の取得
  Game_Bullet.prototype.screenY = function() {
    var th = $gameMap.tileHeight();
    return Math.round($gameMap.adjustY(this._y) * th);
  };

  // 画面 Z 座標の取得
  Game_Bullet.prototype.screenZ = function() {
    return this._z;
  };


  // キャラクターがいる方向（角度）を取得
  Game_Bullet.prototype.angleToCharacter = function(character) {
    var rect = character.collideRect();
    return Math.atan2(rect.centerY() - this._y,
                      rect.centerX() - this._x);
  };

  // プレイヤーがいる方向（角度）を取得
  Game_Bullet.prototype.angleToPlayer = function() {
    return this.angleToCharacter($gamePlayer);
  };


  // フレーム更新
  Game_Bullet.prototype.update = function() {
    var specialShotType = 0;
    if (this._specialWeapon){
      specialShotType = this._specialWeapon.type;
    }
    if (specialShotType == 0){
    }else if(specialShotType == 1){ // バウンド弾
      if (this._swParam > 0){
        var bound = this.isCollideScreen();
        if (bound & 3) this._vy *= -1;
        if (bound & 12) this._vx *= -1;
        if (bound > 0) this._swParam -= 1;
      }
    }else if(specialShotType == 2){ // 追尾弾（速度変化なし方向変化）
      if (this._trackingTarget){ // 対象ロスト判定
        if(!this.canTrack(this._trackingTarget)){
          this._trackingTarget = null;
        }
      }
      if (!this._trackingTarget){ // 追尾対象探索
        this.searchTarget();
      }
      if (this._trackingTarget && this._trachingFrame > 0){ // 追尾中処理
        var rect = this._trackingTarget.collideRect();
        var bx = (rect.centerX() - this._x);
        var by = (rect.centerY() - this._y);
        var ns = this._vx * bx + this._vy * by;
        var gs = this._vx * by - bx * this._vy;
        var ln = Math.sqrt(bx**2 + by**2) * Math.sqrt(this._vx**2 + this._vy**2);
        var rad = Math.acos((ns / ln).clamp(-1,1));
        if (gs < 0) rad = rad - Math.PI;
        rad = rad * (this._trackingPower / 100);
        var vx = this._vx; var vy = this._vy;
        this._vx = Math.cos(rad)*vx - Math.sin(rad)*vy;
        this._vy = Math.sin(rad)*vx + Math.cos(rad)*vy;
        this._trachingFrame -= 1;
      }
    }else if(specialShotType == 3){ // 落下弾
      this._vy += this._swParam;
      this._vy = this._vy.clamp(-3,3);
    }else if(specialShotType == 4){ // 落下地這い弾
      this._vy += this._swParam;
      this._vy = this._vy.clamp(-3,3);
    }else if(specialShotType == 481){ // すやぴ弾
      if($gamePlayer._suyapiShotMode == 2){
        var x = $gamePlayer.collideRect().centerX() + $gamePlayer._shiftFiringX - 1;
        if (this._x + this._vx >= x){ // 着艦開始
          this._vx = x - this._x;
          $gamePlayer.finishSuyapiShot();
          $gamePlayer._suyapiShotMode = 3;
        }
      }else if ($gamePlayer._suyapiShotMode == 3){ // 着艦待ち
        this._vx = 0;
      }else if ($gamePlayer._suyapiShotMode == 4){ // 着艦完了
        this.erase();
        $gamePlayer._suyapiShotMode = 0;
        return false;
      }
    }
    this._x += this._vx;
    this._y += this._vy;
    if(specialShotType == 4){
      if ($gameMap.balletIsCollidedWithTerrain(this)){
        var vyy = this._vy;
        this._vx = -$gameMap._overrayVx; // 地面にめり込んでいる場合速度0にして
        this._vy = 0;
        for(var i = 0; $gameMap.balletIsCollidedWithTerrain(this) && i < 0.3+vyy; i+= 0.1){
          this._y -= 0.1; // 今フレームでのY座標移動が-0.3となるか、地面から出るまで0.1ずつ上昇
        }
      }
      if (this._vx == -$gameMap._overrayVx && !$gameMap.balletIsCollidedWithTerrain(this)){
        this._vx = this._swParam2; // 横移動速度が0で、地面のめりこみが解除された場合、移動速度を再設定
      }
    }
    this._count--;
    if (this._count <= 0 || this.updateCollide()) {
      // すやぴ弾は消える前に特殊処理
      if ($gamePlayer._suyapiShotMode == 1 && specialShotType == 481){
        this._x = 1;
        $gamePlayer._suyapiShotMode = 2;
        return true;
      }
      this.erase();
      return false;
    }
    return true;
  };

  // 削除
  Game_Bullet.prototype.erase = function() {
    this._count = 0;
    this._opacity = 0;
  };

  // 弾によるダメージ処理
  Game_Bullet.prototype.executeDamage = function(character) {
    if (character.isShotInvincible() || character._suyapiShotMode) return;
    if (this._ownerId < 0) {
      if (this._ownerId === -1) {
        var owner = $gamePlayer;
      } else {
        var index = -this._ownerId - 2;
        var owner = $gamePlayer.followers().follower(index);
      }
    } else if (this._ownerId > 0) {
      var owner = $gameMap.event(this._ownerId);
    } else {
      var owner = null;
    }
    var ownerBattler = owner ? owner.battler() : null;
    var target = character.battler();
    if (ownerBattler && target) {
      ownerBattler.clearActions();
      var action = new Game_Action(ownerBattler);
      action.setSkill(this._skillId);
      ownerBattler.setAction(0, action);
      ownerBattler.currentAction().apply(target);
      if (target.result().isHit()) {
        var animeId = action.item().meta.bulletAnime;
        if (animeId) character.requestAnimation(+animeId);
        character.setShotInvincible();
        if (character==$gamePlayer){
          $gamePlayer.battlerDamage();
        }
        if (character != $gamePlayer){
          $gamePlayer.addHitMpRecover(); // 敵に当てると、MP回復予約を追加
        }
      }
      if (target.result().isStateAdded(target.deathStateId())) {
        character.battlerDead();
        if (character != $gamePlayer){
          $gamePlayer.battler().setMp($gamePlayer.battler().mp + 1); // 1匹倒すと1MP回復
        }
      }
      if (Game_CharacterBase.prototype.popupDamage) {
        this.showDamagePopup(character, owner);
      }
    }
  };

  // ダメージポップアップの表示
  Game_Bullet.prototype.showDamagePopup = function(character, owner) {
    var result = character.battler().result();
    if (result.isHit()) {
      var damageType = $dataSkills[this._skillId].damage.type;
      if (damageType > 0 && damageType % 2 === 1) {
        character.popupDamage(result.hpDamage, result.critical);
        if (result.drain) {
          owner.popupDamage(-result.hpDamage, false);
        }
      } else {
        character.popupMpDamage(result.mpDamage, result.critical);
        if (result.drain) {
          owner.popupMpDamage(-result.mpDamage, false);
        }
      }
    } else {
      character.popupMiss();
    }
  }
  
  Game_Bullet.prototype.collideRect = function(character) {
    var rect = new Collide_Rect(this._x - this._collideSize, this._y - this._collideSize, this._collideSize * 2, this._collideSize * 2);
    return rect;
  };

  // キャラクターと接触しているかどうかを返す
  Game_Bullet.prototype.isCollidedWithCharacter = function(character) {
    var bRect = this.collideRect();
    var cRect = character.collideRect();
    return bRect.isCollidedWith(cRect);
  };

  // イベントキャラクターと接触しているかどうかを返す
  Game_Bullet.prototype.isCollidedWithEvents = function() {
    var events = $gameMap.events();
    for (var i = 0; i < events.length; i++) {
      var event = events[i];
      if (!event._through && this.isCollidedWithCharacter(event)) {
        if (this._collidedCharacters.indexOf(event.ownerId()) === -1) {
          this.executeDamage(event);
          this._collidedCharacters.push(event.ownerId());
          return !this._penetrate;
        }
      }
    }
    return false;
  };

  // プレイヤーキャラクターと接触しているかどうかを返す
  Game_Bullet.prototype.isCollidedWithPlayerCharacter = function() {
    if (this.isCollidedWithCharacter($gamePlayer)) {
      var battler = $gamePlayer.battler();
      if (battler && battler.isAlive() &&
          this._collidedCharacters.indexOf($gamePlayer.ownerId()) === -1) {
        if($gamePlayer.hasBarrier()){
          $gamePlayer.reduceBarrier();
        }else{
          this.executeDamage($gamePlayer);
        }
        this._collidedCharacters.push($gamePlayer.ownerId());
        return !this._penetrate;
      }
    }

    return false; // とりあえずフォロワーの接触判定は不要とする

    return $gamePlayer.followers().visibleFollowers().some(function(follower) {
      if (this.isCollidedWithCharacter(follower)) {
        if (follower.battler().isAlive() &&
            this._collidedCharacters.indexOf(follower.ownerId()) === -1) {
          this.executeDamage(follower);
          this._collidedCharacters.push(follower.ownerId());
          return !this._penetrate;
        }
      }
      return false;
    }, this);
  };

  // マップと接触しているかどうかを返す
  Game_Bullet.prototype.isCollideMap = function() {
    var x = Math.floor(this._x);
    var y = Math.floor(this._y);
    return !$gameMap.checkPassageBullet(x, y);
  }

  // 画面端と接触しているかどうかを返す 上:1 下:2　左:4　右:8でor
  Game_Bullet.prototype.isCollideScreen = function() {
    var x = this.screenX();
    var y = this.screenY();
    var ret = 0;
    if (y <= 0) ret += 1;
    if (Graphics.height-1 <= y) ret += 2;
    if (x <= 0) ret += 4;
    if (Graphics.width-1 <= x) ret += 8;
    return ret;
  }

  // 弾消去境界と接触しているかどうかを返す 上:1 下:2　左:4　右:8でor
  Game_Bullet.prototype.isCollideEraseBound = function() {
    var x = this.screenX();
    var y = this.screenY();
    var ret = 0;
    if (y <= -48) ret += 1;
    if (Graphics.height+48-1 <= y) ret += 2;
    if (x <= -48) ret += 4;
    if (Graphics.width+48-1 <= x) ret += 8;
    return ret;
  }

  //-----------------------------------------------------------------------------
  // Game_PlayerBullet
  //

  Game_PlayerBullet.prototype = Object.create(Game_Bullet.prototype);
  Game_PlayerBullet.prototype.constructor = Game_PlayerBullet;

  // 接触判定
  Game_PlayerBullet.prototype.updateCollide = function() {
    if (this._mapCollide && this.isCollideMap()) return true;
    if (this.isCollideEraseBound() > 0 ) return true;
    if (this._friendly) return this.isCollidedWithPlayerCharacter();
    return this.isCollidedWithEvents();
  };

  //-----------------------------------------------------------------------------
  // Game_EnemyBullet
  //

  Game_EnemyBullet.prototype = Object.create(Game_Bullet.prototype);
  Game_EnemyBullet.prototype.constructor = Game_EnemyBullet;

  // 接触判定
  Game_EnemyBullet.prototype.updateCollide = function() {
    if (this._mapCollide && this.isCollideMap()) return true;
    if (this.isCollideEraseBound() > 0 ) return true;
    if (this._friendly) return this.isCollidedWithEvents();
    if ($gamePlayer.isShotInvincible()) return false; // プレイヤー無敵
    return this.isCollidedWithPlayerCharacter();
  };

  //-----------------------------------------------------------------------------
  // Game_DummyBullet
  //

  function Game_DummyBullet() {
      this.initialize.apply(this, arguments);
  }

  Game_DummyBullet.prototype = Object.create(Game_Bullet.prototype);
  Game_DummyBullet.prototype.constructor = Game_DummyBullet;

  // 画面 X 座標の取得
  Game_DummyBullet.prototype.screenX = function() {
    var tw = $gameMap.tileWidth();
    return Math.round(this._x * tw);
  };

  // 画面 Y 座標の取得
  Game_DummyBullet.prototype.screenY = function() {
    var th = $gameMap.tileHeight();
    return Math.round(this._y * th);
  };

  // 接触判定
  Game_DummyBullet.prototype.updateCollide = function() {
    return false;
  };

  //-----------------------------------------------------------------------------
  // Game_CharacterBase
  //

  // メンバ変数の初期化
  var _Game_CharacterBase_initMembers = Game_CharacterBase.prototype.initMembers;
  Game_CharacterBase.prototype.initMembers = function() {
    _Game_CharacterBase_initMembers.call(this);
    this._vx = 0;
    this._vy = 0;
    this._collideW     = 0.375;
    this._collideH     = 0.75;
    this._collideShiftX = 0;
    this._collideShiftY = 0;
    this._clampMode = false;
    this._moveCommandsInSM = [];
    this._commandInSMwait = 0;
  };

  var _Game_CharacterBase_updateStop = Game_CharacterBase.prototype.updateStop;
  Game_CharacterBase.prototype.updateStop = function() {
    _Game_CharacterBase_updateStop.call(this);
    if (TMPlugin.Shooting.SmoothMode) {
      this.updateMove();
    }
  };

  var _Game_CharacterBase_update = Game_CharacterBase.prototype.update;
  Game_CharacterBase.prototype.update = function() {
    _Game_CharacterBase_update.call(this);
    if (this._shotInvincible == null) this._shotInvincible = 0;
    if (this._shotInvincible > 0) this._shotInvincible--;
  };

  // 無敵状態かどうかを返す
  Game_Character.prototype.isShotInvincible = function() {
    if (this._shotInvincible == null) this._shotInvincible = 0;
    return this._shotInvincible > 0;
  };

  // 無敵時間のセット
  Game_Character.prototype.setShotInvincible = function() {
    var battler = this.battler();
    if (battler) {
      var shotParams = battler.shotParams();
      this._shotInvincible = shotParams.invincible;
    }
  };

  // 移動中？
  var _Game_CharacterBase_isMoving = Game_CharacterBase.prototype.isMoving;
  Game_CharacterBase.prototype.isMoving = function() {
    if (TMPlugin.Shooting.SmoothMode ) {
      return false; // SmoothMode時は常にSTOP状態として扱う。これによって常時移動ルート変更を受け付けたり、updateMoveのタイミングを制御する。
    }else{
      return _Game_CharacterBase_isMoving.call(this);
    }
  };

  // 移動の処理
  var _Game_CharacterBase_updateMove = Game_CharacterBase.prototype.updateMove;
  Game_CharacterBase.prototype.updateMove = function() {
    if (TMPlugin.Shooting.SmoothMode) {
      this.updateMoveInSmoothMode();
    }else{
      _Game_CharacterBase_updateMove.call(this);
    }
  };

  // 当たり判定のRECTを取得
  Game_CharacterBase.prototype.collideRect = function() {
    var rect = new Collide_Rect(this._realX + 0.5 - this._collideW + this._collideShiftX,
                                this._realY + 1 - this._collideH + this._collideShiftY,
                                this._collideW * 2, this._collideH);
    return rect;
  };

  // シューティングモードの時の移動処理
  Game_CharacterBase.prototype.updateMoveInSmoothMode = function() {
    if ($gameMessage.isBusy()){return;}
    this.processMoveCommandInSmoothMode();
    if (this._vx != 0){
      this._realX += this._vx;
      if (!this.isThrough() && this._clampMode){
        var rect = this.collideRect();
        this._realX = this._realX.clamp($gameMap._displayX - (rect._left - this._realX), $gameMap._displayX+17 - (rect._left + rect._width - this._realX)); // 横着
      }
      this._x = Math.floor(this._realX);
    }
    if (this._vy != 0){
      this._realY += this._vy;
      if (!this.isThrough() && this._clampMode && !this._suyapiShotMode){ // すやぴショット中はクランプしない
        var rect = this.collideRect();
        this._realY = this._realY.clamp($gameMap._displayY - (rect._top - this._realY), $gameMap._displayY+12 - (rect._top + rect._height - this._realY)); // 横着
      }
      this._y = Math.floor(this._realY);
    }
  };

  // シューティングモードの時の移動コマンド処理
  Game_CharacterBase.prototype.processMoveCommandInSmoothMode = function() {
    if (this._commandInSMwait > 0){
      this._commandInSMwait -= 1;
      return;
    }
    while(true){
      var params = this._moveCommandsInSM.shift();
      if (!params) return; // コマンドがない場合は抜ける。
      var command = params.shift();
      switch (command){
        case "WAIT":
          this._commandInSMwait = params[0] - 1;
          return; // WAITの場合は抜ける。その他はすべて連続処理。
        case "SET_VELOCITY":
          this._vx = params[0];
          this._vy = params[1];
          break;
        case "SET_VELOCITY_BY_POLAR":
          this._vx = params[0] * Math.cos(Math.PI * params[1] / 180);
          this._vy = params[0] * Math.sin(Math.PI * params[1] / 180);
          break;
        case "HERMITE":
          this.culcHermiteIP(params);
          break;
        case "HERMITE_BY_POLAR":
          var s_r = params[0] * Math.cos(Math.PI * params[1] / 180);
          var s_angle = params[0] * Math.sin(Math.PI * params[1] / 180);
          var f_r = params[4] * Math.cos(Math.PI * params[5] / 180);
          var f_angle = params[4] * Math.sin(Math.PI * params[5] / 180);
          params[0] = s_r; params[1] = s_angle; params[4] = f_r; params[5] = f_angle;
          this.culcHermiteIP(params);
          break;
      }
    }
  };

  // 移動先予約
  Game_CharacterBase.prototype.移動クリア = function(frames) {
    this._moveCommandsInSM = [];
  };
  Game_CharacterBase.prototype.移動ウェイト = function(frames) {
    this._moveCommandsInSM.push(["WAIT",frames]);
  };
  Game_CharacterBase.prototype.速度変更 = function(vx,vy,wait=1) {
    this._moveCommandsInSM.push(["SET_VELOCITY",vx,vy]);
    this.移動ウェイト(wait);
  };
  Game_CharacterBase.prototype.極座標速度で速度変更 = function(r,angle,wait=1) {
    this._moveCommandsInSM.push(["SET_VELOCITY_BY_POLAR",r,angle]);
    this.移動ウェイト(wait);
  };
  Game_CharacterBase.prototype.エルミート = function(s_vx,s_vy,tx,ty,f_vx,f_vy,frames,set_fv = false) {
    this._moveCommandsInSM.push(["HERMITE",s_vx,s_vy,tx,ty,f_vx,f_vy,frames]);
    if(set_fv) this.移動ウェイト(1);// その後にコマンドが続いても終点位置での速度設定を最低1フレーム持続させる場合はset_fv=trueにする
  };
  Game_CharacterBase.prototype.極座標速度でエルミート = function(s_r,s_angle,tx,ty,f_r,f_angle,frames,set_fv = false) {
    this._moveCommandsInSM.push(["HERMITE_BY_POLAR",s_r,s_angle,tx,ty,f_r,f_angle,frames]);
    if(set_fv) this.移動ウェイト(1);// その後にコマンドが続いても終点位置での速度設定を最低1フレーム持続させる場合はset_fv=trueにする
  };
  Game_CharacterBase.prototype.正弦波 = function(vx,frame,length,amp=1){
    var dt = (length * Math.PI) / frame;
    var ny = 0;
    for(var t = dt; t<= length * Math.PI; t += dt){
      var y = amp * Math.sin(t);
      this._moveCommandsInSM.push(["SET_VELOCITY",vx,y - ny]);
      this.移動ウェイト(1);
      ny = y;
    }
  }
  Game_CharacterBase.prototype.放物線 = function(vx,height,frame){
    var ay = 2*height / (((frame-1)/2) ** 2);
    var vy = -ay * (frame-1) / 2;
    for(var t = 0; t< frame; t += 1){
      this._moveCommandsInSM.push(["SET_VELOCITY",vx,vy]);
      this.移動ウェイト(1);
      vy += ay;
    }
  }

  Game_CharacterBase.prototype.culcHermiteIP = function(param) {
    var sp = [this._realX,this._realY]; // 始点位置
    var sv = [param[0],param[1]];       // 始点速度ベクトル
    var fp = [param[2],param[3]];       // 終点位置
    var fv = [param[4],param[5]];       // 終点速度ベクトル
    var frames = param[6];
  
    var move_root = [];
    if (frames == 0){
      move_root.push(fp);
    }else{
      for(var i=1; i<=frames; i++){
        var t = (i / frames);
        var root_x = (  2*sp[0]+  sv[0]-2*fp[0]+fv[0] ) * t**3 +
                     ( -3*sp[0]-2*sv[0]+3*fp[0]-fv[0] ) * t**2 +
                     sv[0] * t +
                     sp[0];
        var root_y = (  2*sp[1]+  sv[1]-2*fp[1]+fv[1] ) * t**3 +
                     ( -3*sp[1]-2*sv[1]+3*fp[1]-fv[1] ) * t**2 +
                     sv[1] * t +
                     sp[1];
        move_root.push([root_x,root_y])
      }
    }
    this._moveCommandsInSM.unshift(["SET_VELOCITY",fv[0],fv[1]]);
    while(move_root.length > 0){
      var moveTo = move_root.pop();
      var moveFrom = null;
      if (move_root.length <= 0){
        moveFrom = sp;
      }else{
        moveFrom = move_root[move_root.length-1];
      }
      this._moveCommandsInSM.unshift(["WAIT",1]);
      this._moveCommandsInSM.unshift(["SET_VELOCITY",moveTo[0]-moveFrom[0],moveTo[1]-moveFrom[1]]);
    }
  }

  // 消去境界と接触しているかどうかを返す 上:1 下:2　左:4　右:8でor
  Game_CharacterBase.prototype.isCollideEraseBound = function() {
    var x = this.screenX();
    var y = this.screenY();
    var ret = 0;
    if (y <= -48) ret += 1;
    if (Graphics.height+48-1 <= y) ret += 2;
    if (x <= -48) ret += 4;
    if (Graphics.width+48-1 <= x) ret += 8;
    return ret;
  }

  //-----------------------------------------------------------------------------
  // Game_Character
  //

  // メンバ変数の初期化
  var _Game_Character_initMembers = Game_Character.prototype.initMembers;
  Game_Character.prototype.initMembers = function() {
    _Game_Character_initMembers.call(this);
    this._shiftFiringY = 0;
    this._shiftFiringX = 0;
    this._shotDelay    = 0;
    this._shotCount    = 0;
  };

  // イベントIDを返す（TMBossHpGauge.jsとの連携のため）
  Game_Character.prototype.eventId = function() {
    return -1;
  };

  // キャラクターのいる方向（角度）を取得
  Game_Character.prototype.angleToCharacter = function(character) {
    var cr = character.collideRect();
    var tr = this.collideRect();
    var x = tr.centerX() + this._shiftFiringX;
    var y = tr.centerY() + this._shiftFiringY - (this.shiftY() ) / $gameMap.tileHeight();
    return Math.atan2(cr.centerY() - y, cr.centerX() - x);
  };

  // プレイヤーのいる方向（角度）を取得
  Game_Character.prototype.angleToPlayer = function() {
    return this.angleToCharacter($gamePlayer);
  };
  
  // ｎ方向ショット
  Game_Character.prototype.nwayShot = function(n, space, angle, speed, count,
                                               type, index, skillId, xShift = 0) {
    var ownerId = this.ownerId();
    if (ownerId < 0) {
      if (!$gameSystem.isPlayerShotEnabled()) return false;
    } else {
      if (!$gameSystem.isEnemyShotEnabled()) return false;
    }
    angle = angle - (space * (n - 1) / 2);
    var rect = this.collideRect();
    var x = rect.centerX() + this._shiftFiringX + xShift;
    var y = rect.centerY() + this._shiftFiringY - (this.shiftY() ) / $gameMap.tileHeight();
    for (var i = 0; i < n; i++) {
      $gameMap.addBullet(x, y, 200 + i, Math.cos(angle) * speed,
                         Math.sin(angle) * speed, angle, count, type, index,
                         skillId, ownerId, this.battler().specialWeapon());
      angle += space;
    }
    return true;
  };

  // 自機狙いｎ方向ショット
  Game_Character.prototype.nwayAim = function(n, space, angle, speed, count,
                                              type, index, skillId) {
    var a = angle + this.angleToPlayer();
    this.nwayShot(n, space, a, speed, count, type, index, skillId);
  };

  // 全方位ショット
  Game_Character.prototype.nallShot = function(n, angle, speed, count, type,
                                               index, skillId) {
    var ownerId = this.ownerId();
    if (ownerId < 0) {
      if (!$gameSystem.isPlayerShotEnabled()) return;
    } else {
      if (!$gameSystem.isEnemyShotEnabled()) return;
    }
    var space = Math.PI * 2 / n;
    var rect = this.collideRect();
    var x = rect.centerX() + this._shiftFiringX;
    var y = rect.centerY() + this._shiftFiringY - (this.shiftY() ) / $gameMap.tileHeight();
    for (var i = 0; i < n; i++) {
      $gameMap.addBullet(x, y, 200 + i, Math.cos(angle) * speed,
                         Math.sin(angle) * speed, angle, count, type, index,
                         skillId, ownerId, this.battler().specialWeapon());
      angle += space;
    }
  };

  // 自機狙い全方位ショット
  Game_Character.prototype.nallAim = function(n, angle, speed, count, type,
                                              index, skillId) {
    var a = angle + this.angleToPlayer()
    this.nallShot(n, a, speed, count, type, index, skillId);
  };
  
  // プレイヤー（フォロワー）のショット処理
  Game_Character.prototype.executeShot = function() {
    var battler = this.battler();
    if (battler && battler.canMove()) {
      var shotParams = battler.shotParams();
      var skill = $dataSkills[shotParams.skillId];
      if ( shotParams.skillId!=1 && !battler.canPaySkillCost(skill) ) return; // コスト支払い可能でなければダメ
      battler.paySkillCost(skill); // コストも払いましょう
      if (shotParams.way > 0) {
        var angle = this._shotAngle == null ? this.shotAngleFromDirection() : this._shotAngle;
        this._shotDelay = shotParams.interval;
        if (shotParams.burst > 0){
          this._shotCount += 1;
          if (this._shotCount >= shotParams.burst){
            this._shotCount = 0;
            this._shotDelay += shotParams.burstInterval;
          }
        }
        var ret = this.nwayShot(shotParams.way, shotParams.space, angle, shotParams.speed,
                             shotParams.count, shotParams.type, shotParams.index,
                             shotParams.skillId);
        if (ret){
          battler.paySkillCost(skill); // コストも払いましょう
        }
        return ret;
      }
    }
    return false;
  };

  // 向いている方向から弾発射角度を返す
  Game_Character.prototype.shotAngleFromDirection = function() {
    if (Imported.SAN_AnalogMove) {
      return -this.analogMove()._directionRadian - Math.PI / 2;
    } else {
      if (this._direction === 2) {
        return 1.570796;
      } else if (this._direction === 4) {
        return Math.PI;
      } else if (this._direction === 6) {
        return 0;
      } else {
        return 4.712388;
      }
    }
  };

  // 発射方向を現在向いている方向に固定する
  Game_Character.prototype.holdShotAngle = function(release) {
    this._shotAngle = release ? null : this.shotAngleFromDirection();
  };

  // バトラーが戦闘不能になったときの処理
  Game_Character.prototype.battlerDead = function() {
    var animeId = this._deadAnime || TMPlugin.Shooting.DefaultDeadAnimeId;
    if (animeId) this.requestAnimation(animeId);
  };


  //-----------------------------------------------------------------------------
  // Game_Player
  //

  // メンバ変数の初期化
  var _Game_Player_initMembers = Game_Player.prototype.initMembers;
  Game_Player.prototype.initMembers = function() {
    _Game_Player_initMembers.call(this);
    this._autoShot = false;
    this._noScroll = false;
    this._moveLog = [];
    this._barrier = 0;
    this._hitMpRecover = 0;
  };

  // 王様の初期化（シューティングステージの再スタート時に呼ぶ）
  Game_Player.prototype.resetQueenState = function() {
    var ids = $gameParty.allMembers().map(function(actor) {return actor.actorId();});
    ids.forEach( function(id) {$gameParty.removeActor(id);});
    $gameParty.addActor(1);
    this._moveLog = [];
    var battler = this.battler();
    if(battler){
      battler.recoverAll();
      battler.setMp(0);
      battler.forceChangeEquip(0, $dataWeapons[1]);
    }
    $gameParty.loseGold(99999999);
    this._barrier = 0;
    this._hitMpRecover = 0;
    this._suyapiShotMode = 0;
    this._doubleTriggerCount = 0;
  };

var HitMpRecoverInterval = 10;
  Game_Player.prototype.addHitMpRecover = function() {
    if (this._hitMpRecover > HitMpRecoverInterval*9) return;
    this._hitMpRecover += HitMpRecoverInterval;
  };

  Game_Player.prototype.updateHitMpRecover = function() {
    if (this._hitMpRecover < 1) return;
    this._hitMpRecover -= 1;
    if (this._hitMpRecover % HitMpRecoverInterval == 0) this.battler().setMp(this.battler().mp + 1);
  };

  Game_Player.prototype.hasBarrier = function() {
    return this._barrier;
  };

  Game_Player.prototype.reduceBarrier = function() {
    this._barrier -= 1;
    this._barrier = this._barrier.clamp(0,9999);
    if (TMPlugin.Shooting.BarrierSe) {
      AudioManager.playSe(TMPlugin.Shooting.BarrierSe);
    }
  };

  Game_Player.prototype.setBarrier = function(num) {
    this._barrier = num;
    this._barrier = this._barrier.clamp(0,9999);
  };

  // オート射撃のセット
  Game_Player.prototype.setAutoShot = function(autoShot) {
    this._autoShot = autoShot;
  };

  // バトラーを返す
  Game_Player.prototype.battler = function() {
    return $gameParty.leader();
  };

  // オーナーIDを返す
  Game_Player.prototype.ownerId = function() {
    return -1;
  };

  // 向き固定状態かどうかを返す
  var _Game_Player_isDirectionFixed = Game_Player.prototype.isDirectionFixed;
  Game_Player.prototype.isDirectionFixed = function() {
    if (this.isMoveRouteForcing()) return _Game_Player_isDirectionFixed.call(this);
    if (this._shotHold == null) this._shotHold = false;
    return this._shotHold;
  };

  // フレーム更新
  var _Game_Player_update = Game_Player.prototype.update;
  Game_Player.prototype.update = function(sceneActive) {
    if (sceneActive) {
      this.holdByInput();
      this.shotByInput();
      this.updateHitMpRecover();
    }
    _Game_Player_update.call(this, sceneActive);
//debug[todo]外す
//if (Input.isPressed('shift')) this.setBarrier(2);
  };

  // ボタン入力による向き固定
  Game_Player.prototype.holdByInput = function() {
    if (TMPlugin.Shooting.HoldType === 'SWITCH') {
      if (Input.isTriggered('hold')) {
        this._shotHold = !this._shotHold;
        this.holdShotAngle(!this._shotHold);
        this._followers.holdShotAngle(!this._shotHold);
      }
    } else {
      var lastShotHold = this._shotHold;
      this._shotHold = Input.isPressed('hold');
      if (this._shotHold !== lastShotHold) {
        this.holdShotAngle(!this._shotHold);
        this._followers.holdShotAngle(!this._shotHold);
      }
    }
  };

  // ボタン入力による攻撃
  Game_Player.prototype.shotByInput = function() {
    if (this._suyapiShotMode) return; // すやぴ弾中
//    var shotInput = Input.isPressed('shot') || this._autoShot;
    var shotInput = Input.isPressed('ok') || this._autoShot || TouchInput.isPressed();
    if (this._shotDelay > 0) {
      this._shotDelay--;
    } else if (shotInput) {
      if (this.executeShot()) {
        if (TMPlugin.Shooting.LeaderShotSe) {
          AudioManager.playSe(TMPlugin.Shooting.LeaderShotSe);
        }
        // ここにセリフ用の関数がお邪魔しますよ
        $gameMap.reserveAttackShoutAuto();
      }
    }
    this._followers.shotByInput(shotInput);

  };

// 小数点以下の座標がほしいので横着してここに書いておきます
Game_Map.prototype.canvasToMapX2 = function(x) {
    var tileWidth = this.tileWidth();
    var originX = this._displayX * tileWidth;
    var mapX = ((originX + x) / tileWidth);
    return this.roundX(mapX);
};

Game_Map.prototype.canvasToMapY2 = function(y) {
    var tileHeight = this.tileHeight();
    var originY = this._displayY * tileHeight;
    var mapY = ((originY + y) / tileHeight);
    return this.roundY(mapY);
};

  // 方向ボタン入力による移動処理
  var _Game_Player_moveByInput = Game_Player.prototype.moveByInput;
  Game_Player.prototype.moveByInput = function() {
    if (TMPlugin.Shooting.SmoothMode) {
      if(this.canMove()){
        this._vx = 0;
        this._vy = 0;
        var speed = this.distancePerFrame() * 2;
        if (Input.isPressed('left')) {
          this._vx = -speed;
        } else if (Input.isPressed('right')) {
          this._vx = speed;
        }
        if (Input.isPressed('up')) {
          this._vy = -speed;
        } else if (Input.isPressed('down')) {
          this._vy = speed;
        }
        if (this._vx != 0 && this._vy != 0){
          this._vx /= Math.sqrt(2);
          this._vy /= Math.sqrt(2);
        }
        if (TouchInput.isPressed()){
          var targetX = $gameMap.canvasToMapX2(TouchInput.x);
          var targetY = $gameMap.canvasToMapY2(TouchInput.y);
          this._vx = (targetX - this._realX);
          this._vy = (targetY - this._realY);
          var len = Math.sqrt(this._vx**2 + this._vy**2);
          if (len > speed){
            this._vx = this._vx * speed / len;
            this._vy = this._vy * speed / len;
          }
        }
        if ($gamePlayer.pinchLevel() >= 3) this._vx += 0.2*(-0.5 + Math.random());
        if ($gamePlayer.pinchLevel() >= 3) this._vy += 0.2*(-0.5 + Math.random());
        if ($gamePlayer.pinchLevel() >= 2) this._vx *= 2*Math.random();
        if ($gamePlayer.pinchLevel() >= 2) this._vy *= 2*Math.random();
        if (this._suyapiShotMode){ // すやぴ弾中
          this._vx = 0;
          this._vy = 0;
        }
      }
    }else{
      _Game_Player_moveByInput.call(this);
    }
  };

  // シューティングモードの時の移動処理
  Game_Player.prototype.updateMoveInSmoothMode = function() {
    Game_Character.prototype.updateMoveInSmoothMode.call(this);
    if (this.haveOption() && (this._gatheringMinipi || !this._moveLog[0] || this._moveLog[0].toString() != [this._realX, this._realY].toString())){
      this.unshiftMoveLog();
    }
  };

  Game_Player.prototype.haveOption = function() {
    var followers = this.followers();
    for(var i = 0;i<$gameParty.maxBattleMembers() - 1;i++){
      var follower = followers.follower(i);
      if (follower && follower.smallSuyapiId()) { return true ;}
    }
    return false;
  };

  var _Game_Player_updateDashing = Game_Player.prototype.updateDashing;
  Game_Player.prototype.updateDashing = function() {
    if (TMPlugin.Shooting.SmoothMode) {
/*
      if (!this.isInVehicle() && !$gameMap.isDashDisabled()) {
        this._dashing = this.isDashButtonPressed() || $gameTemp.isDestinationValid();
      } else {
        this._dashing = false;
      }
*/
      this._dashing = false;
    }else{
      _Game_Player_updateDashing.call(this);
    }
  };

  
  // リフレッシュ
  var _Game_Player_refresh = Game_Player.prototype.refresh;
  Game_Player.prototype.refresh = function() {
    _Game_Player_refresh.call(this);
    var actor = $gameParty.leader();
    if (actor) {
      var data = actor.actor();
      this._collideW     = +(data.meta.cw || 0.375);
      this._collideH     = +(data.meta.ch || 0.75);
      this._collideShiftX     = +(data.meta.shiftcx || 0);
      this._collideShiftY     = +(data.meta.shiftcy || 0);
      this._shiftFiringY = +(data.meta.shiftFiringY || 0);
      this._shiftFiringX = +(data.meta.shiftFiringX || 0);
      this._deadAnime    = +(data.meta.deadAnime || 0);
    }
  };

  var _Game_Player_characterName = Game_Player.prototype.characterName;
  Game_Player.prototype.characterName = function() {
    var actor = $gameParty.leader();
    if (actor && actor.isDead()) {
      var deadCharacterName = actor.deadCharacterName();
      if (deadCharacterName) return deadCharacterName;
    }
    return _Game_Player_characterName.call(this);
  };

  var _Game_Player_characterIndex = Game_Player.prototype.characterIndex;
  Game_Player.prototype.characterIndex = function() {
    var actor = $gameParty.leader();
    if (actor && actor.isDead()) {
      var deadCharacterIndex = actor.deadCharacterIndex();
      if (deadCharacterIndex !== null) return deadCharacterIndex;
    }
    return _Game_Player_characterIndex.call(this);
  };

  // バトラーが戦闘不能になったときの処理
  Game_Player.prototype.battlerDead = function() {
    Game_Character.prototype.battlerDead.call(this);
    if (TMPlugin.Shooting.PlayerDeadEventId) {
      $gameTemp.reserveCommonEvent(TMPlugin.Shooting.PlayerDeadEventId);
    }
  };

  // バトラーがダメージを受けたときの処理
  Game_Player.prototype.battlerDamage = function() {
    if (TMPlugin.Shooting.PlayerDamageEventId) {
      $gameTemp.reserveCommonEvent(TMPlugin.Shooting.PlayerDamageEventId);
      $gameMap.interruptShout("被弾");
      $gameMap.reserveShout("初回");
    }
  };

  var _Game_Player_updateScroll = Game_Player.prototype.updateScroll;
  Game_Player.prototype.updateScroll = function(lastScrolledX, lastScrolledY) {
    if (this._noScroll){
      
    }else{
      _Game_Player_updateScroll.call(this);
    }
  };

  // イベントとの衝突判定
  Game_Player.prototype.isCollidedWith = function(character) {
    if (this.eventId() === character.eventId()) return false;
    tr = this.collideRect();
    cr = character.collideRect();
    return tr.isCollidedWith(cr);
  };
  Game_Player.prototype.collide_events = function() {
    var ret = [];
    var gp = this;
    this.collideTargets().forEach( function(event) {
      if (gp.isCollidedWith(event)) ret.push(event);
    });
    return ret;
  };
  // 衝突判定を行う対象を返す
  Game_Player.prototype.collideTargets = function() {
    return $gameMap.events();
  };

  // デフォルトのイベント起動まわりはSmoothModeの時は殺しておく
  var _Game_Player_checkEventTriggerHere = Game_Player.prototype.checkEventTriggerHere;
  Game_Player.prototype.checkEventTriggerHere = function(triggers) {
    if (TMPlugin.Shooting.SmoothMode) {
    }else{
      _Game_Player_checkEventTriggerHere.call(this,triggers);
    }
  };
  var _Game_Player_checkEventTriggerThere = Game_Player.prototype.checkEventTriggerThere;
  Game_Player.prototype.checkEventTriggerThere = function(triggers) {
    if (TMPlugin.Shooting.SmoothMode) {
    }else{
      _Game_Player_checkEventTriggerThere.call(this,triggers);
    }
  };
  var _Game_Player_checkEventTriggerTouch = Game_Player.prototype.checkEventTriggerTouch;
  Game_Player.prototype.checkEventTriggerTouch = function(x,y) {
    if (TMPlugin.Shooting.SmoothMode) {
    }else{
      _Game_Player_checkEventTriggerTouch.call(this,x,y);
    }
  };

  Game_Player.prototype.pinchLevel = function(){
    if (!$gamePlayer.battler()) return 0;
    return Math.max(0,(4 - $gamePlayer.battler().hp));
  };

  Game_Player.prototype.moveLog = function(num) {
    if( !this._moveLog || this._moveLog.length == 0) return [this._realX, this._realY];
    return this._moveLog[Math.min(num,this._moveLog.length - 1)];
  };

  Game_Player.prototype.unshiftMoveLog = function() {
    this._moveLog.unshift([this._realX, this._realY]);
    if(this._moveLog.length > 100) this._moveLog.pop();
  };

  Game_Player.prototype.gatherMinipi = function(b) {
    this._gatheringMinipi = b;
  };

  //-----------------------------------------------------------------------------
  // Game_Party
  //

  Game_Party.prototype.maxBattleMembers = function() {
    return 15; // 4人とかケチくせぇこと言うんじゃねぇぜ！！　このぐらいだ！
  };

  //-----------------------------------------------------------------------------
  // Game_Follower
  //

  // バトラーを返す
  Game_Follower.prototype.battler = function() {
    return this.actor();
  };

  // オーナーIDを返す
  Game_Follower.prototype.ownerId = function() {
    return -this._memberIndex - 1;
  };

  var _Game_Follower_refresh = Game_Follower.prototype.refresh;
  Game_Follower.prototype.refresh = function() {
    _Game_Follower_refresh.call(this);
    var actor = this.actor();
    if (actor) {
      var data = actor.actor();
      this._collideW     = +(data.meta.cw || 0.375);
      this._collideH     = +(data.meta.ch || 0.75);
      this._collideShiftX     = +(data.meta.shiftcx || 0);
      this._collideShiftY     = +(data.meta.shiftcy || 0);
      this._shiftFiringY = +(data.meta.shiftFiringY || 0);
      this._shiftFiringX = +(data.meta.shiftFiringX || 0);
      this._deadAnime    = +(data.meta.deadAnime || 0);
      this._smallSuyapiId = +(data.meta.smallSuyapiId || 0);
    }
  };

  Game_Follower.prototype.shotByInput = function(shotInput) {
    if (this._shotDelay > 0) {
      this._shotDelay--;
    } else if (shotInput) {
      this.executeShot();
    }
  };
  
  var _Game_Follower_characterName = Game_Follower.prototype.characterName;
  Game_Follower.prototype.characterName = function() {
    var actor = this.actor();
    if (actor && actor.isDead()) {
      var deadCharacterName = actor.deadCharacterName();
      if (deadCharacterName) {
        return deadCharacterName;
      }
    }
    return _Game_Follower_characterName.call(this);
  };

  var _Game_Follower_characterIndex = Game_Follower.prototype.characterIndex;
  Game_Follower.prototype.characterIndex = function() {
    var actor = this.actor();
    if (actor && actor.isDead()) {
      var deadCharacterIndex = actor.deadCharacterIndex();
      if (deadCharacterIndex !== null) {
        return deadCharacterIndex;
      }
    }
    return _Game_Follower_characterIndex.call(this);
  };

  var _Game_Follower_update = Game_Follower.prototype.update;
  Game_Follower.prototype.update = function() {
    _Game_Follower_update.call(this);
    if (TMPlugin.Shooting.SmoothMode) {
      this.copyPosition($gamePlayer);
      this._through = true;
      if(this.smallSuyapiId()){
        var log = $gamePlayer.moveLog(this.smallSuyapiId() * 20);
        this.setPosition(log[0],log[1]);
      }
    }
  };

  Game_Follower.prototype.smallSuyapiId = function() {
    if (this._smallSuyapiId == 0) return null;
    return this._smallSuyapiId;
  };

  //-----------------------------------------------------------------------------
  // Game_Followers
  //

  Game_Followers.prototype.holdShotAngle = function(release) {
    this.forEach(function(follower) {
        follower.holdShotAngle(release);
    }, this);
  };

  Game_Followers.prototype.shotByInput = function(shotInput) {
    this.forEach(function(follower) {
        follower.shotByInput(shotInput);
    }, this);
  };



  //-----------------------------------------------------------------------------
  // Game_Event
  //

var ScreenOutTimeOut = 120;
  // メンバ変数の初期化
  var _Game_Event_initMembers = Game_Event.prototype.initMembers;
  Game_Event.prototype.initMembers = function() {
    _Game_Event_initMembers.call(this);
    this._enemyId = 0;
    this._battler = null;
    this._deadSelfSwitch = null;
    this._screenOutTime = 0;
  };

  var _Game_Event_update = Game_Event.prototype.update;
  Game_Event.prototype.update = function() {
    _Game_Event_update.call(this);
    var key = [$gameMap.mapId(), this.eventId(), TMPlugin.Shooting.ClearSwitch];
    if(this.battler() && $gameSelfSwitches.value(key) == false){
      if(this.isCollideEraseBound()>0){
        this._screenOutTime += 1;
        if(this._screenOutTime >= ScreenOutTimeOut){
          $gameSelfSwitches.setValue(key, true);
          this._screenOutTime = 0;
        }
      }else{
        this._screenOutTime = 0;
      }
    }
  };

  // バトラーを返す
  Game_Event.prototype.battler = function() {
    return this._battler;
  };

  // オーナーIDを返す
  Game_Event.prototype.ownerId = function() {
    return this._eventId;
  };

  // イベントページのセットアップ
  var _Game_Event_setupPage = Game_Event.prototype.setupPage;
  Game_Event.prototype.setupPage = function() {
    _Game_Event_setupPage.call(this);
    if (this._pageIndex >= 0) {
      // ページ変更時はスムーズモード移動速度および移動コマンドを初期化する
        this._vx = 0;
        this._vy = 0;
        this._clampMode = false;
        this._moveCommandsInSM = [];
        this._commandInSMwait = 0;
      // エネミーデータ関係の初期化
      var data = this.event();
      this._enemyId = +(data.meta.enemy || 0);
      this._collideW = +(data.meta.cw || 0.375);// アイテム用に当たり判定のみエネミーでなくともmetaがあれば適用
      this._collideH = +(data.meta.ch || 0.75); // アイテム用に当たり判定のみエネミーでなくともmetaがあれば適用
      if (this._enemyId > 0) {
        this._battler = new Game_Enemy(this._enemyId, -this.eventId(), 0);
        this._collideShiftX     = +(data.meta.shiftcx || 0);
        this._collideShiftY     = +(data.meta.shiftcy || 0);
        this._shiftFiringY = +(data.meta.shiftFiringY || 0);
        this._shiftFiringX = +(data.meta.shiftFiringX || 0);
        var enemy = this._battler.enemy();
        this._deadAnime = +(enemy.meta.deadAnime || 0);
      } else {
        this._battler = null;
      }
    }
  };

  // 無敵時間のセット
  Game_Event.prototype.setShotInvincible = function() {
  };

  // バトラーが戦闘不能になったときの処理
  Game_Event.prototype.battlerDead = function() {
    this.gainRewards();
    if (TMPlugin.Shooting.DeadSwitch) {
      var key = [$gameMap.mapId(), this._eventId, TMPlugin.Shooting.DeadSwitch];
      $gameSelfSwitches.setValue(key, true);
    }
    $gameMap.reserveBeatShoutAuto();
    Game_Character.prototype.battlerDead.call(this);
  };
  
  Game_Event.prototype.gainRewards = function() {
    this.gainExp();
    this.gainGold();
    var items = this.battler().makeDropItems();
    for (var i = 0; i < items.length; i++) {
      this.gainRewardItem(items[i]);
    }
  };
  
  Game_Event.prototype.gainExp = function() {
    var exp = this.battler().exp()
    if (exp > 0) {
      if (Imported['GetInformation']) {
        CommonPopupManager._popEnable = CommonPopupManager.popEnable();
      }
      $gameParty.allMembers().forEach(function(actor) {
        actor.gainExp(exp);
      });
      if (Imported['GetInformation']) {
        CommonPopupManager._popEnable = false;
      }
    }
  };
  
  Game_Event.prototype.gainGold = function() {
    gold = this.battler().gold();
    if (gold > 0) {
      if (Imported['GetInformation']) {
        CommonPopupManager._popEnable = CommonPopupManager.popEnable();
      }
      $gameParty.gainGold(gold * (4 ** $gamePlayer.pinchLevel()) * ($gameParty.hasGoldDouble() ? 2 : 1) );
      if (Imported['GetInformation']) {
        CommonPopupManager._popEnable = false;
      }
    }
  };

  Game_Event.prototype.gainRewardItem = function(item) {
    if (Imported['GetInformation']) {
      CommonPopupManager._popEnable = CommonPopupManager.popEnable();
    }
    $gameParty.gainItem(item, 1);
    if (Imported['GetInformation']) {
      CommonPopupManager._popEnable = false;
    }
  };

  // デフォルトのイベント接触起動まわりはSmoothModeの時は殺しておく
  var _Game_Event_checkEventTriggerTouch = Game_Event.prototype.checkEventTriggerTouch;
  Game_Event.prototype.checkEventTriggerTouch = function(triggers) {
    if (TMPlugin.Shooting.SmoothMode) {
    }else{
      _Game_Event_checkEventTriggerTouch.call(this,triggers);
    }
  };

  // 巨大キャラに都合が悪いので指定したら無視するように。
  var _Game_Event_isNearTheScreen = Game_Event.prototype.isNearTheScreen;
  Game_Event.prototype.isNearTheScreen = function() {
    if (this.event().meta.ignoreNearScreen) return true;
    return _Game_Event_isNearTheScreen.call(this);
  };


  //-----------------------------------------------------------------------------
  // Game_Interpreter
  //

  var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function(command, args) {
    _Game_Interpreter_pluginCommand.call(this, command, args);
    if (command === 'startAutoShot') {
      $gamePlayer.setAutoShot(true);
    } else if (command === 'stopAutoShot') {
      $gamePlayer.setAutoShot(false);
    } else if (command === 'nwayShot') {
      var character = this.character(0);
      var battler = character.battler();
      if (battler.canMove()) {
        character.nwayShot.apply(character, args.map(Number));
      }
    } else if (command === 'nwayAim') {
      var character = this.character(0);
      var battler = character.battler();
      if (battler.canMove()) {
        character.nwayAim.apply(character, args.map(Number));
      }
    } else if (command === 'nallShot') {
      var character = this.character(0);
      var battler = character.battler();
      if (battler.canMove()) {
        character.nallShot.apply(character, args.map(Number));
      }
    } else if (command === 'nallAim') {
      var character = this.character(0);
      var battler = character.battler();
      if (battler.canMove()) {
        character.nallAim.apply(character, args.map(Number));
      }
    } else if (command === 'stopPlayerShot') {
      $gameSystem.setPlayerShotEnabled(false, args[0]);
    } else if (command === 'startPlayerShot') {
      $gameSystem.setPlayerShotEnabled(true, args[0]);
    } else if (command === 'stopEnemyShot') {
      $gameSystem.setEnemyShotEnabled(false, args[0]);
    } else if (command === 'startEnemyShot') {
      $gameSystem.setEnemyShotEnabled(true, args[0]);
    } else if (command === 'deletePlayerBullets') {
      $gameMap.clearPlayerBullets();
    } else if (command === 'deleteEnemyBullets') {
      $gameMap.clearEnemyBullets();
    } else if (command === 'forceShot') {
      if (args[0] == null) {
        $gamePlayer.executeShot();
        $gamePlayer.followers().forEach(function(follower) {
          follower.executeShot();
        }, null);
      } else {
        if (+args[0] === 0) {
          $gamePlayer.executeShot();
        } else {
          $gamePlayer.followers().follower(+args[0] - 1).executeShot();
        }
      }
    } else if (command === 'shootingMode') {// ゲームモード変更
      TMPlugin.Shooting.SmoothMode = (args[0]=='true');
    } else if (command === 'noScroll') {// スクロールなし
      $gamePlayer._noScroll = (args[0]=='true');
    } else if (command === 'clampMode') {// 画面端で移動停止モード
      $gamePlayer._clampMode = (args[0]=='true');
    } else if (command === 'allEnemyClear') {// 全エネミーをクリア
      $gameMap.events().forEach(function(event) {
        if(event._enemyId == 0 && !event.event().meta.item) return; // アイテムも含むように
        var key = [$gameMap.mapId(), event.eventId(), TMPlugin.Shooting.ClearSwitch];
        $gameSelfSwitches.setValue(key, true);
      });
    } else if (command === 'allEnemyStop') {// 全エネミーを停止（ゲームオーバー用）
      $gameMap.events().forEach(function(event) {
        event._vx = 0;
        event._vy = 0;
        event._moveCommandsInSM = [];
      });
    } else if (command === 'allEnemyKill') {// 生存中の全エネミーを破壊
      $gameMap.events().forEach(function(event) {
        var battler = event.battler();
        var key = [$gameMap.mapId(), event.eventId(), TMPlugin.Shooting.DeadSwitch];
        var key2 = [$gameMap.mapId(), event.eventId(), TMPlugin.Shooting.ClearSwitch];
        if(!battler || !battler.isAlive() || $gameSelfSwitches.value(key)==true || $gameSelfSwitches.value(key2)==true ) return;
        event.battlerDead();
      });
    } else if (command === 'popEnemy') {// 指定イベントIDのエネミーを指定座標にpop
      anum = args.map(Number);
      var key = [$gameMap.mapId(), anum[0], TMPlugin.Shooting.DeadSwitch];
      var key2 = [$gameMap.mapId(), anum[0], TMPlugin.Shooting.ClearSwitch];
      var events = $gameMap._events;
      // 念のため一旦消す
      $gameSelfSwitches.setValue(key, true);
      $gameSelfSwitches.setValue(key2, true);
      events[args[0]].refresh();
      // 復活
      $gameSelfSwitches.setValue(key, false);
      $gameSelfSwitches.setValue(key2, false);
      events[args[0]].setPosition(anum[1],anum[2]);
      events[args[0]].refresh(); // ここでリフレッシュしてしまおう
    } else if (command === 'equipSpecialWeapon') {// 指定イベントID（イベントID=0の時は自身）のイベントに特殊弾(特殊弾ID=0は外す)を装備
      anum = args.map(Number);
      var events = $gameMap._events;
      var battler = this.character(0).battler();
      if(args[0]!=0) battler = events[args[0]].battler();
      anum.shift();
      if (battler) battler.equipSpecialWeapon.call(battler,anum);
    } else if (command === 'resetQueenState') { // 王様初期化
      $gamePlayer.resetQueenState();
    } else if (command === 'setQueenBarrier') { // 王様バリア
      anum = args.map(Number);
      $gamePlayer.setBarrier(anum[0]);
    }
  };

  //-----------------------------------------------------------------------------
  // Sprite_Character
  //

  var _Sprite_Character_updateOther = Sprite_Character.prototype.updateOther;
  Sprite_Character.prototype.updateOther = function() {
    _Sprite_Character_updateOther.call(this);
    this.opacity = this._character.opacity() * (this._character.isShotInvincible() ? (Graphics.frameCount%2 == 0 ? 0.5 : 0) : 1);
    if (this._character == $gamePlayer && $gamePlayer.hasBarrier()){
      if(!this._barrierSprite){
        this._barrierSprite = new Sprite_Barrier();
        this.addChild(this._barrierSprite);
      }
      this._barrierSprite.opacity = 255 * $gamePlayer.hasBarrier() / 2;
    }else if (this._barrierSprite){
      this.removeChild(this._barrierSprite);
      this._barrierSprite = null;
    }
  };

  var _Sprite_Character_characterPatternY = Sprite_Character.prototype.characterPatternY;
  Sprite_Character.prototype.characterPatternY = function() {
    if (TMPlugin.Shooting.SmoothMode && this._character == $gamePlayer && !$gamePlayer._suyapiShotMode){
      return (this._character._vy > 0 ? 0 : (this._character._vy < 0 ? 3 : 2)); // 横向きシューティング用
    }else{
      return _Sprite_Character_characterPatternY.call(this);
    }
  };

  //-----------------------------------------------------------------------------
  // Sprite_Barrier
  //

var BarrierAnimationInterval = 30;
  function Sprite_Barrier() {
    this.initialize.apply(this, arguments);
  }

  Sprite_Barrier.prototype = Object.create(Sprite_Base.prototype);
  Sprite_Barrier.prototype.constructor = Sprite_Barrier;

  Sprite_Barrier.prototype.initialize = function() {
    Sprite_Base.prototype.initialize.call(this);
    this.bitmap = ImageManager.loadSystem("barrier");
    this._barrierFrameCount = 0;
    this.update();
  };

  Sprite_Barrier.prototype.update = function() {
    Sprite_Base.prototype.update.call(this);
    var h = this.bitmap.height;
    var w = h;
    var maxPattern = this.bitmap.width / w;
    this._barrierFrameCount += 1;
    var pattern = Math.floor(this._barrierFrameCount/BarrierAnimationInterval);
    if (pattern >= maxPattern) {this._barrierFrameCount = 0; pattern = 0;}
    this.setFrame(w * pattern, 0, w, h);
    this.x = -w/2 + 8;
    this.y = -h/2 - 96;
  };

  //-----------------------------------------------------------------------------
  // Sprite_Bullet
  //

  function Sprite_Bullet() {
    this.initialize.apply(this, arguments);
  }

  Sprite_Bullet.prototype = Object.create(Sprite.prototype);
  Sprite_Bullet.prototype.constructor = Sprite_Bullet;

  // 初期化
  Sprite_Bullet.prototype.initialize = function(bullet) {
    Sprite.prototype.initialize.call(this);
    this.anchor.x = 0.5;
    this.anchor.y = 0.5;
    this._bullet = bullet;
    this._bulletName  = '';
    this._bulletIndex = 0;
  };

  // フレーム更新
  Sprite_Bullet.prototype.update = function() {
    Sprite.prototype.update.call(this);
    this.opacity = this._bullet.opacity();
    if (this.opacity > 0) {
      this.updateBitmap();
      this.x = this._bullet.screenX();
      this.y = this._bullet.screenY();
      this.z = this._bullet.screenZ();
      this.rotation = this._bullet.angle();
    }
  };

  // 転送元ビットマップの更新
  Sprite_Bullet.prototype.updateBitmap = function() {
    if (this._bulletName !== this._bullet.bulletName() ||
        this._bulletIndex !== this._bullet.bulletIndex()) {
      this._bulletName = this._bullet.bulletName();
      this._bulletIndex = this._bullet.bulletIndex();
      this.setBulletBitmap();
    }
  };

  // ビットマップの設定
  Sprite_Bullet.prototype.setBulletBitmap = function() {
    this.bitmap = ImageManager.loadSystem(this._bulletName);
    if (this.bitmap.width === 0) {
      this._bulletName = '';
    } else {
      var pw = Math.floor(this.bitmap.width / 8);
      var sx = this._bulletIndex % 8 * pw;
      var sy = Math.floor(this._bulletIndex / 8) * pw;
      this.setFrame(sx, sy, pw, pw);
      this.blendMode = TMPlugin.Shooting.BulletBlendTable[this._bullet.type()];
    }
  };

  //-----------------------------------------------------------------------------
  // Sprite_EquipDummy
  //

  function Sprite_EquipDummy() {
    this.initialize.apply(this, arguments);
  }

  Sprite_EquipDummy.prototype = Object.create(Sprite.prototype);
  Sprite_EquipDummy.prototype.constructor = Sprite_EquipDummy;

  Sprite_EquipDummy.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.anchor.x = 0.5;
    this.anchor.y = 1;
    this.x = TMPlugin.Shooting.EquipDummyX;
    this.y = TMPlugin.Shooting.EquipDummyY;
    this._actor = null;
    this._animationCount = 0;
    this._pattern = 0;
    this._shotShiftY = 0;
  };
  
  Sprite_EquipDummy.prototype.setActor = function(actor) {
    this._actor = actor;
    this._shotShiftY = +(actor.actor().meta.ch || 0.75) / 2;
    this._shiftFiringY = +(actor.actor().meta.shiftFiringY || 0);
    this._shiftFiringX = +(actor.actor().meta.shiftFiringX || 0);
  };

  Sprite_EquipDummy.prototype.update = function() {
    Sprite.prototype.update.call(this);
    this.updateBitmap();
    this.updateCharacterFrame();
    this.updateAnimation();
  };
  
  Sprite_EquipDummy.prototype.updateBitmap = function() {
    if (this.isImageChanged()) {
      this._characterName = this._actor.characterName();
      this._characterIndex = this._actor.characterIndex();
      this.setCharacterBitmap();
    }
  };
  
  Sprite_EquipDummy.prototype.isImageChanged = function() {
    return (this._characterName !== this._actor.characterName() ||
            this._characterIndex !== this._actor.characterIndex());
  };

  Sprite_EquipDummy.prototype.setCharacterBitmap = function() {
    this.bitmap = ImageManager.loadCharacter(this._characterName);
    this._isBigCharacter = ImageManager.isBigCharacter(this._characterName);
  };
  
  Sprite_EquipDummy.prototype.updateCharacterFrame = function() {
    var pw = this.patternWidth();
    var ph = this.patternHeight();
    var sx = (this.characterBlockX() + this.characterPatternX()) * pw;
    var sy = (this.characterBlockY() + this.characterPatternY()) * ph;
    this.setFrame(sx, sy, pw, ph);
  };

  Sprite_EquipDummy.prototype.characterBlockX = function() {
    if (this._isBigCharacter) {
      return 0;
    } else {
      var index = this._actor.characterIndex();
      return index % 4 * 3;
    }
  };

  Sprite_EquipDummy.prototype.characterBlockY = function() {
    if (this._isBigCharacter) {
      return 0;
    } else {
      var index = this._actor.characterIndex();
      return Math.floor(index / 4) * 4;
    }
  };

  Sprite_EquipDummy.prototype.characterPatternX = function() {
    return this._pattern === 3 ? 1 : this._pattern;
  };

  Sprite_EquipDummy.prototype.characterPatternY = function() {
    return 0;
  };

  Sprite_EquipDummy.prototype.patternWidth = function() {
    return this.bitmap.width / (this._isBigCharacter ? 3 : 12);
  };

  Sprite_EquipDummy.prototype.patternHeight = function() {
    return this.bitmap.height / (this._isBigCharacter ? 4 : 8);
  };

  Sprite_EquipDummy.prototype.updateAnimation = function() {
    this.updateAnimationCount();
    if (this._animationCount >= 18) {
      this.updatePattern();
      this._animationCount = 0;
    }
  };
  
  Sprite_EquipDummy.prototype.updateAnimationCount = function() {
    this._animationCount += 1.5;
  };

  Sprite_EquipDummy.prototype.updatePattern = function() {
    this._pattern = (this._pattern + 1) % 4;
  };

  Sprite_EquipDummy.prototype.shotX = function() {
    return (this.x  + this._shiftFiringX)/ $gameMap.tileWidth();
  };
  
  Sprite_EquipDummy.prototype.shotY = function() {
    return (this.y  + this._shiftFiringY) / $gameMap.tileHeight() - this._shotShiftY;
  };
  
  //-----------------------------------------------------------------------------
  // Spriteset_Destination
  //

  var _Sprite_Destination_updatePosition = Sprite_Destination.prototype.updatePosition;
  Sprite_Destination.prototype.updatePosition = function() {
    if (Imported.SAN_AnalogMove) {
      var tileWidth = $gameMap.tileWidth();
      var tileHeight = $gameMap.tileHeight();
      var x = $gamePlayer.analogMove()._targRealX;
      var y = $gamePlayer.analogMove()._targRealY;
      this.x = $gameMap.adjustX(x) * tileWidth;
      this.y = $gameMap.adjustY(y) * tileHeight;
    } else {
      _Sprite_Destination_updatePosition.call(this);
    }
  };

  //-----------------------------------------------------------------------------
  // Spriteset_Map
  //

  var _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
  Spriteset_Map.prototype.createLowerLayer = function() {
    _Spriteset_Map_createLowerLayer.call(this);
    this.createBullets();
  };

  // 弾スプライトの作成
  Spriteset_Map.prototype.createBullets = function() {
    this._bulletSprites = [];
    $gameMap.playerBullets().forEach(function(bullet) {
      this._bulletSprites.push(new Sprite_Bullet(bullet));
    }, this);
    $gameMap.enemyBullets().forEach(function(bullet) {
      this._bulletSprites.push(new Sprite_Bullet(bullet));
    }, this);
    for (var i = 0; i < this._bulletSprites.length; i++) {
      this._baseSprite.addChild(this._bulletSprites[i]);
    }
    this._dummyPlayerSprite = new Sprite_Dummy_Character($gamePlayer);
    this._baseSprite.addChild(this._dummyPlayerSprite);
  };

  //-----------------------------------------------------------------------------
  // Sprite_Dummy_Character (プレイヤーキャラクターを一時的に弾の前に表示するためのダミー)
  //

function Sprite_Dummy_Character() {
    this.initialize.apply(this, arguments);
}

Sprite_Dummy_Character.prototype = Object.create(Sprite_Character.prototype);
Sprite_Dummy_Character.prototype.constructor = Sprite_Dummy_Character;

Sprite_Dummy_Character.prototype.initialize = function(character) {
    Sprite_Character.prototype.initialize.call(this,character);
};

Sprite_Dummy_Character.prototype.updateOther = function() {
    Sprite_Character.prototype.updateOther.call(this);
    this.opacity = this._character._dummyVisible ? 255 : 0;
};

  //-----------------------------------------------------------------------------
  // Window_Option
  //

  var _Window_Option_makeCommandList = Window_Options.prototype.makeCommandList;
  Window_Options.prototype.makeCommandList = function() {
    _Window_Option_makeCommandList.call(this);
    if (TMPlugin.Shooting.PadConfigCommand) {
      this.addCommand(TMPlugin.Shooting.PadConfigCommand, 'padConfig');
    }
  };

  var _Window_Options_statusText = Window_Options.prototype.statusText;
  Window_Options.prototype.statusText = function(index) {
    var symbol = this.commandSymbol(index);
    if (symbol === 'padConfig') {
      return '';
    } else {
      return _Window_Options_statusText.call(this, index);
    }
  };

  Window_Options.prototype.isPadSymbol = function(symbol) {
    return symbol.contains('padButton');
  };

  var _Window_Options_processOk = Window_Options.prototype.processOk;
  Window_Options.prototype.processOk = function() {
    var index = this.index();
    var symbol = this.commandSymbol(index);
    if (symbol === 'padConfig') {
      this.playOkSound();
      this.updateInputData();
      this.deactivate();
      this.callHandler('padConfig');
    } else {
      _Window_Options_processOk.call(this);
    }
  };

  var _Window_Options_cursorRight = Window_Options.prototype.cursorRight;
  Window_Options.prototype.cursorRight = function(wrap) {
    var index = this.index();
    var symbol = this.commandSymbol(index);
    if (symbol !== 'padConfig') {
      _Window_Options_cursorRight.call(this, wrap);
    }
  };

  var _Window_Options_cursorLeft = Window_Options.prototype.cursorLeft;
  Window_Options.prototype.cursorLeft = function(wrap) {
    var index = this.index();
    var symbol = this.commandSymbol(index);
    if (symbol !== 'padConfig') {
      _Window_Options_cursorLeft.call(this, wrap);
    }
  };

  //-----------------------------------------------------------------------------
  // Window_PadOptions
  //

  function Window_PadOptions() {
    this.initialize.apply(this, arguments);
  }

  Window_PadOptions.prototype = Object.create(Window_Options.prototype);
  Window_PadOptions.prototype.constructor = Window_PadOptions;

  Window_PadOptions.prototype.initialize = function() {
    Window_Options.prototype.initialize.call(this, 0, 0);
    this.hide();
    this.deactivate();
  };

  Window_PadOptions.prototype.makeCommandList = function() {
    for (var i = 1; i <= 12; i++) {
      this.addCommand('パッドボタン' + i, 'padButton' + i);
    }
  };

  Window_PadOptions.prototype.statusWidth = function() {
    return 120;
  };

  Window_PadOptions.prototype.statusText = function(index) {
    var value = this.getConfigValue(index);
    if (value) {
      return TMPlugin.Shooting.PadButtonNames[TMPlugin.Shooting.PadButtons.indexOf(value)];
    } else {
      return '';
    }
  };

  Window_PadOptions.prototype.processOk = function() {
    var index = this.index();
    var value = this.getConfigValue(index);
    value = TMPlugin.Shooting.PadButtons.indexOf(value);
    value += 1;
    if (value >= TMPlugin.Shooting.PadButtons.length) {
      value = 0;
    }
    this.changeValue(index, TMPlugin.Shooting.PadButtons[value]);
  };

  Window_PadOptions.prototype.cursorRight = function(wrap) {
    var index = this.index();
    var value = this.getConfigValue(index);
    value = TMPlugin.Shooting.PadButtons.indexOf(value);
    value = (value + 1).clamp(0, TMPlugin.Shooting.PadButtons.length - 1);
    this.changeValue(index, TMPlugin.Shooting.PadButtons[value]);
  };

  Window_PadOptions.prototype.cursorLeft = function(wrap) {
    var index = this.index();
    var value = this.getConfigValue(index);
    value = TMPlugin.Shooting.PadButtons.indexOf(value);
    value = (value - 1).clamp(0, TMPlugin.Shooting.PadButtons.length - 1);
    this.changeValue(index, TMPlugin.Shooting.PadButtons[value]);
  };

  Window_PadOptions.prototype.changeValue = function(index, value) {
    var lastValue = this.getConfigValue(index);
    if (lastValue !== value) {
      this.setConfigValue(index, value);
      this.redrawItem(index);
      SoundManager.playCursor();
    }
  };

  Window_PadOptions.prototype.getConfigValue = function(index) {
    return ConfigManager.getPadButton(index);
  };

  Window_PadOptions.prototype.setConfigValue = function(index, value) {
    ConfigManager.setPadButton(index, value);
  };

  //-----------------------------------------------------------------------------
  // Scene_Base
  //

  var _Scene_Base_checkGameover = Scene_Base.prototype.checkGameover;
  Scene_Base.prototype.checkGameover = function() {
    if (TMPlugin.Shooting.UseGameover) {
      _Scene_Base_checkGameover.call(this);
    }
  };

  //-----------------------------------------------------------------------------
  // Scene_Equip
  //

  var _Scene_Equip_create = Scene_Equip.prototype.create;
  Scene_Equip.prototype.create = function() {
    this.createDummy();
    this.createBullets();
    _Scene_Equip_create.call(this);
    this.addChild(this._dummySprite);
    for (var i = 0; i < this._bulletSprites.length; i++) {
      this.addChild(this._bulletSprites[i]);
    }
  };
  
  Scene_Equip.prototype.createDummy = function() {
    this._dummySprite = new Sprite_EquipDummy();
    this._shotDelay = 0;
  };
  
  Scene_Equip.prototype.createBullets = function() {
    this._playerBullets = [];
    this._alivePlayerBullets = [];
    this._blankPlayerBullets = [];
    for (var i = 0; i < TMPlugin.Shooting.MaxPlayerBullet; i++) {
      this._playerBullets.push(new Game_DummyBullet());
      this._blankPlayerBullets.push(i);
    }
    this._bulletSprites = [];
    this._playerBullets.forEach(function(bullet) {
      this._bulletSprites.push(new Sprite_Bullet(bullet));
    }, this);
  };

  Scene_Equip.prototype.nwayShot = function(n, space, angle, speed, count,
                                            type, index, skillId) {
    angle = angle - (space * (n - 1) / 2);
    var x = this._dummySprite.shotX();
    var y = this._dummySprite.shotY();
    for (var i = 0; i < n; i++) {
      this.addBullet(x, y, 200 + i, Math.cos(angle) * speed,
                         Math.sin(angle) * speed, angle, count, type, index,
                         skillId, -1);
      angle += space;
    }
  };

  Scene_Equip.prototype.addBullet = function(x, y, z, vx, vy, angle, count, type,
                                          index, skillId, ownerId) {
    if (this._blankPlayerBullets.length > 0) {
      var bulletIndex = this._blankPlayerBullets.shift();
      this._playerBullets[bulletIndex].setup(x, y, z, vx, vy, angle, count,
                                             type, index, skillId, ownerId);
      this._alivePlayerBullets.push(bulletIndex);
    }
  };

  var _Scene_Equip_refreshActor = Scene_Equip.prototype.refreshActor;
  Scene_Equip.prototype.refreshActor = function() {
    _Scene_Equip_refreshActor.call(this);
    this._dummySprite.setActor(this.actor());
    this.clearPlayerBullets();
    this._shotDelay = 0;
  };

  var _Scene_Equip_update = Scene_Equip.prototype.update;
  Scene_Equip.prototype.update = function() {
    _Scene_Equip_update.call(this);
    this.updateBullets();
    this.updateShot();
  };

  Scene_Equip.prototype.updateBullets = function() {
    for (var i = this._alivePlayerBullets.length - 1; i >= 0; i--) {
      var index = this._alivePlayerBullets[i];
      if (!this._playerBullets[index].update()) {
        this._alivePlayerBullets.splice(i, 1);
        this._blankPlayerBullets.push(index);
      }
    }
  };
  
  Scene_Equip.prototype.updateShot = function() {
    if (this._shotDelay > 0) {
      this._shotDelay--;
    } else {
      if (this._itemWindow.active && this._statusWindow._tempActor) {
        var shotParams = this._statusWindow._tempActor.shotParams();
      } else {
        var shotParams = this.actor().shotParams();
      }
      if (shotParams.way > 0) {
        var angle = Math.PI / 2;
        this.nwayShot(shotParams.way, shotParams.space, angle, shotParams.speed,
                      shotParams.count, shotParams.type, shotParams.index,
                      shotParams.skillId);
        this._shotDelay = shotParams.interval;
      }
    }
  };
  
  Scene_Equip.prototype.clearPlayerBullets = function() {
    this._alivePlayerBullets.forEach(function(index) {
      this._playerBullets[index].erase();
    }, this);
    this._blankPlayerBullets.concat(this._alivePlayerBullets);
    this._alivePlayerBullets = [];
  };

  //-----------------------------------------------------------------------------
  // Scene_Options
  //

  var _Scene_Options_create = Scene_Options.prototype.create;
  Scene_Options.prototype.create = function() {
    _Scene_Options_create.call(this);
    this.createPadOptionsWindow();
  };
  
  var _Scene_Options_createOptionsWindow = Scene_Options.prototype.createOptionsWindow;
  Scene_Options.prototype.createOptionsWindow = function() {
    _Scene_Options_createOptionsWindow.call(this);
    this._optionsWindow.setHandler('padConfig', this.onPadConfig.bind(this));
  };
  
  Scene_Options.prototype.createPadOptionsWindow = function() {
    this._padOptionsWindow = new Window_PadOptions();
    this._padOptionsWindow.setHandler('cancel', this.cancelPadConfig.bind(this));
    this.addWindow(this._padOptionsWindow);
  };
  
  Scene_Options.prototype.onPadConfig = function() {
    this._optionsWindow.hide();
    this._padOptionsWindow.show();
    this._padOptionsWindow.activate();
  };
  
  Scene_Options.prototype.cancelPadConfig = function() {
    this._padOptionsWindow.hide();
    this._optionsWindow.show();
    this._optionsWindow.activate();
  };

})();

//////////////////////////////////////////////////////////////////////////////////////
// このへんからオーバーレイマップ表示用改造（別プラグインにしてもいいけど面倒なので）
//////////////////////////////////////////////////////////////////////////////////////

  //-----------------------------------------------------------------------------
  // DataManager
  //

  var $dataOverrayMap = null;

  DataManager.loadOverrayDataFile = function(name, src) {
    var xhr = new XMLHttpRequest();
    var url = 'data/' + src;
    xhr.open('GET', url);
    xhr.overrideMimeType('application/json');
    xhr.onload = function() {
        if (xhr.status < 400) {
            window[name] = JSON.parse(xhr.responseText);
            DataManager.onLoad(window[name]);
        }
    };
    xhr.onerror = this._overrayMapLoader || function() {
        DataManager._errorUrl = DataManager._errorUrl || url;
    };
    window[name] = null;
    xhr.send();
  };

  DataManager.loadOverrayMapData = function(mapId) {
    if (mapId == 0){
      $dataOverrayMap = null;
    }else{
      var filename = 'Map%1.json'.format(mapId.padZero(3));
      this._overrayMapLoader = ResourceHandler.createLoader('data/' + filename, this.loadDataFile.bind(this, '$dataOverrayMap', filename));
      this.loadOverrayDataFile('$dataOverrayMap', filename);
    }
  };

  //-----------------------------------------------------------------------------
  // Game_Interpreter
  //

  Game_Interpreter.prototype.オーバーレイマップ位置 = function(x,y){
    $gameMap.setPosOverrayMap(x,y);
  }

  Game_Interpreter.prototype.オーバーレイマップ速度 = function(x,y){
    $gameMap.setVelocityOverrayMap(x,y);
  }

  Game_Interpreter.prototype.オーバーレイマップ表示 = function(wait = false) {
    if (wait){
      while(!$dataOverrayMap){
      }
    };
    $gameMap.showOverrayMap();
  }

  Game_Interpreter.prototype.オーバーレイマップ非表示 = function() {
    $gameMap.hideOverrayMap();
  }

  Game_Interpreter.prototype.オーバーレイマップロード = function(mapId) {
    $gameMap.loadOverrayMap(mapId);
  }

  Game_Interpreter.prototype.オーバーレイマップ転送設定 = function(x,y,destX,destY) {
    $gameMap.setTransferOverrayMap(x,y,destX,destY);
  }

  Game_Interpreter.prototype.オーバーレイマップ転送設定解除 = function(x,y) {
    $gameMap.deleteTransferOverrayMap(x,y);
  }

  Game_Interpreter.prototype.オーバーレイマップイベント設定 = function(x,y,commonEventId,overrayEventId) {
    $gameMap.setEventOverrayMap(x,y,commonEventId,overrayEventId);
  }

  //-----------------------------------------------------------------------------
  // Game_Map
  //

  // セットアップ
  var _overray_Game_Map_setup = Game_Map.prototype.setup;
  Game_Map.prototype.setup = function(mapId) {
    _overray_Game_Map_setup.call(this, mapId);
    DataManager.loadOverrayMapData(0); // マップ移動ごとにオーバーレイは消去（したほうが便利よね？）
    this._eventOMSetting = null;
    this._overrayX = 0;
    this._overrayY = 0;
    this._overrayVx = 0;
    this._overrayVy = 0;
    this._overrayVisible = false; // 読み込み完了するまでは表示しないほうが吉
    this._overrayMapInterpreter = [];
  };

  Game_Map.prototype.overrayWidth = function() {
    return $dataOverrayMap.width;
  };

  Game_Map.prototype.overrayHeight = function() {
    return $dataOverrayMap.height;
  };

  Game_Map.prototype.overrayData = function() {
    if (!$dataOverrayMap) return null;
    return $dataOverrayMap.data;
  };

  Game_Map.prototype.isOverrayLoopHorizontal = function() {
    return $dataOverrayMap.scrollType === 2 || $dataOverrayMap.scrollType === 3;
  };

  Game_Map.prototype.isOverrayLoopVertical = function() {
    return $dataOverrayMap.scrollType === 1 || $dataOverrayMap.scrollType === 3;
  };

  Game_Map.prototype.loadOverrayMap = function(mapId) {
    DataManager.loadOverrayMapData(mapId);
  }

  Game_Map.prototype.setPosOverrayMap = function(x,y) {
    this._overrayX = x;
    this._overrayY = y;
    if (!$dataOverrayMap) return;
    this._overrayX = this._overrayX.clamp(0, this.overrayWidth()); // とりあえずループ考慮せず
    this._overrayY = this._overrayY.clamp(0, this.overrayHeight()); // とりあえずループ考慮せず
  }

  Game_Map.prototype.setVelocityOverrayMap = function(x,y) {
    this._overrayVx = x;
    this._overrayVy = y;
  }

  Game_Map.prototype.overrayX = function() {
    return this._overrayX;
  }

  Game_Map.prototype.overrayY = function() {
    return this._overrayY;
  }

  Game_Map.prototype.showOverrayMap = function() {
    this._overrayVisible = true;
  }

  Game_Map.prototype.hideOverrayMap = function() {
    this._overrayVisible = false;
  }

  Game_Map.prototype.isOverrayVisible = function() {
    return this._overrayVisible;
  }

  Game_Map.prototype.overrayTileset = function() {
    return $dataTilesets[$dataOverrayMap.tilesetId];
  };

  Game_Map.prototype.overrayTilesetFlags = function() {
    var tileset = this.overrayTileset();
    if (tileset) {
      return tileset.flags;
    } else {
      return [];
    }
  };

  Game_Map.prototype.layeredOverrayTiles = function(x, y) {
    var tiles = [];
    for (var i = 0; i < 4; i++) {
      tiles.push(this.overrayTileId(x, y, 3 - i));
    }
    return tiles;
  };

  Game_Map.prototype.overrayTileId = function(x, y, z) {
    var width = $dataOverrayMap.width;
    var height = $dataOverrayMap.height;
    return $dataOverrayMap.data[(z * height + y) * width + x] || 0;
  };

Game_Map.prototype.updateOverrayMapInterpreter = function() {
    if (!this._overrayMapInterpreter) return;
    var nextArray = [];
    this._overrayMapInterpreter.forEach(function(interpreter){
        if (interpreter.isRunning()){
            interpreter.update();
            nextArray.push(interpreter);
        }
    });
    this._overrayMapInterpreter = nextArray;
};

  // フレーム更新
  var _overray_Game_Map_update = Game_Map.prototype.update;
  Game_Map.prototype.update = function(sceneActive) {
    _overray_Game_Map_update.call(this, sceneActive);
    this.updateOverrayMap();
    this.updatePlayerCollideWithTerrain();
    this.updateOverrayMapInterpreter();
  };

  Game_Map.prototype.updateOverrayMap = function() {
    if (DataManager._errorUrl) {
      throw new Error('Failed to load: ' + DataManager._errorUrl);
    }
    this._overrayX = this._overrayX + this._overrayVx;
    this._overrayY = this._overrayY + this._overrayVy;
    if (!$dataOverrayMap) return; // ここからはマップ読み込み完了後のみ可。
    if(!this._eventOMSetting ) this.setTransferOverrayMapFirst(); // 端到達時のマップ転送を設定
    // 転送設定の確認と転送
    var x = Math.floor(this._overrayX);
    var y = Math.floor(this._overrayY);
    var overrayMapEvent = this._eventOMSetting[[x,y]];
    if(overrayMapEvent){
      var kind = overrayMapEvent[0];
      if(kind == "TRANSFER"){
        this._overrayX = overrayMapEvent[1] + (this._overrayX - x);
        this._overrayY = overrayMapEvent[2] + (this._overrayY - y);
      }else if (kind == "COMMON_EVENT"){
        // コモンイベント起動
        var event = $dataCommonEvents[overrayMapEvent[1]];
        if (event) {
          var interpreter = new Game_Interpreter();
          interpreter.setup(event.list);
          this._overrayMapInterpreter.push(interpreter);
        }
        $gameVariables.setValue(TMPlugin.Shooting.OMEventIDVariable, overrayMapEvent[2]); // もっとマシなやり方はあるけど、とりあえず現状はこれで動くのでOK
//        $gameVariables.setValue(TMPlugin.Shooting.OMEventIDVariable, x*1000+y); // 位置バージョン
        this._eventOMSetting[[x,y]] = false; // とりあえずイベントは一回起動したら消す（何度も起動したい場合は再度登録してもらう）
      }
    }
    // clamp
    this._overrayX = this._overrayX.clamp(0, this.overrayWidth()); // とりあえずループ考慮せず
    this._overrayY = this._overrayY.clamp(0, this.overrayHeight()); // とりあえずループ考慮せず
  }

  // プレイヤーのオーバーレイ地形との接触判定とその処理
  Game_Map.prototype.updatePlayerCollideWithTerrain = function() {
    if ($gameMap.isEventRunning()) return; // イベント起動時は無視
    if ($gamePlayer.isShotInvincible()) return; // 無敵時はダメージ系は無視
    var battler = $gamePlayer.battler();
    if (battler && !battler.isAlive()) return; // 死んでたら無視
    if (this.isCollidedWithTerrain($gamePlayer)){
      // ダメージ
      var dummyBullet = new Game_EnemyBullet();
      dummyBullet.setup(0, 0, 0, 0, 0, 0, 0, 0, 0, TMPlugin.Shooting.WallHitSkillId, $gamePlayer.ownerId());
      dummyBullet.executeDamage($gamePlayer);
    }
  };

  // オーバーレイ地形との衝突判定
  Game_Map.prototype.isCollidedWithTerrain = function( character ) {
    if (!$dataOverrayMap) return false;
    rect = character.collideRect();
    rect._left = rect._left + character.scrolledX() - character._realX;
    rect._top  = rect._top  + character.scrolledY() - character._realY;

    var ltTerrainX = Math.floor(rect._left + this.overrayX());
    var ltTerrainY = Math.floor(rect._top  + this.overrayY());
    var rbTerrainX = Math.floor(rect._left + rect._width + this.overrayX());
    var rbTerrainY = Math.floor(rect._top + rect._height + this.overrayY());
    // キャラクターの当たり判定が含まれているオーバーレイ地形のマスについて通行可否を見て
    // 一つでも通行不可なら衝突と判定。
    for(var y = ltTerrainY; y <= rbTerrainY; y++ ){
      for(var x = ltTerrainX; x <= rbTerrainX; x++ ){
        if(!this.checkPassageOverray(x,y,(1 << (6 / 2 - 1)) & 0x0f)){
          return true;
        }
      }
    }
    return false;
  };

  // オーバーレイ地形と弾の衝突判定
  Game_Map.prototype.balletIsCollidedWithTerrain = function( ballet ) {
    if (!$dataOverrayMap) return false;
    rect = ballet.collideRect();
    rect._left = rect._left + $gameMap.adjustX(ballet._x) - ballet._x;
    rect._top  = rect._top  + $gameMap.adjustX(ballet._y) - ballet._y;

    var ltTerrainX = Math.floor(rect._left + this.overrayX());
    var ltTerrainY = Math.floor(rect._top  + this.overrayY());
    var rbTerrainX = Math.floor(rect._left + rect._width + this.overrayX());
    var rbTerrainY = Math.floor(rect._top + rect._height + this.overrayY());
    // キャラクターの当たり判定が含まれているオーバーレイ地形のマスについて通行可否を見て
    // 一つでも通行不可なら衝突と判定。
    for(var y = ltTerrainY; y <= rbTerrainY; y++ ){
      for(var x = ltTerrainX; x <= rbTerrainX; x++ ){
        if(!this.checkPassageOverray(x,y,(1 << (6 / 2 - 1)) & 0x0f)){
          return true;
        }
      }
    }
    return false;
  };

  // オーバーレイ地形が通行可能か？
  Game_Map.prototype.checkPassageOverray = function(x, y, bit) {
    var flags = this.overrayTilesetFlags();
    var tiles = this.layeredOverrayTiles(x, y);
    for (var i = 0; i < tiles.length; i++) {
        var flag = flags[tiles[i]];
        if ((flag & 0x10) !== 0)  // [*] No effect on passage
            continue;
        if ((flag & bit) === 0)   // [o] Passable
            return true;
        if ((flag & bit) === bit) // [x] Impassable
            return false;
    }
    return false;
  };

var TransferHeight = 13;
  Game_Map.prototype.setTransferOverrayMapFirst = function(){
    this._eventOMSetting = {};
    var x = this.overrayWidth() - 17;
    var destX = 0;
    for(var y = 0;y<this.overrayHeight();y += TransferHeight){
      var destY = y + TransferHeight;
      if( this.overrayHeight() <= destY )break;
      this.setTransferOverrayMap(x,y,destX,destY);
    }
  };

  Game_Map.prototype.setTransferOverrayMap = function(x,y,destX,destY){
    if (!this._eventOMSetting) return null;
    this._eventOMSetting[[x,y]] = ["TRANSFER",destX,destY];
  };

  Game_Map.prototype.deleteTransferOverrayMap = function(x,y){
    if (!this._eventOMSetting) return null;
    this._eventOMSetting[[x,y]] = null;
  };

  Game_Map.prototype.setEventOverrayMap = function(x,y,commonEventId,overrayEventId){
    if (!this._eventOMSetting) return null;
    this._eventOMSetting[[x,y]] = ["COMMON_EVENT",commonEventId,overrayEventId];
  };

  //-----------------------------------------------------------------------------
  // Spriteset_Map
  //

  Spriteset_Map.prototype.checkAndCreateOverrayTilemap = function() {
    if(this._overrayData == $gameMap.overrayData()) return;
    if(this._overrayTilemap){
      this._baseSprite.removeChild(this._overrayTilemap);
      this._overrayTilemap = null;
    }
    this._overrayData = $gameMap.overrayData();
    if(!$gameMap.overrayData()) return;
    if (Graphics.isWebGL()) {
      this._overrayTilemap = new ShaderTilemap();
    } else {
      this._overrayTilemap = new Tilemap();
    }
    this._overrayTilemap.tileWidth = $gameMap.tileWidth();// まあ同じでしょ……
    this._overrayTilemap.tileHeight = $gameMap.tileHeight();// まあ同じでしょ……
    this._overrayTilemap.setData($gameMap.overrayWidth(), $gameMap.overrayHeight(), $gameMap.overrayData());
    this._overrayTilemap.horizontalWrap = $gameMap.isOverrayLoopHorizontal();
    this._overrayTilemap.verticalWrap = $gameMap.isOverrayLoopVertical();
    this.loadOverrayTileset();
    this._baseSprite.addChild(this._overrayTilemap);
  };

  Spriteset_Map.prototype.loadOverrayTileset = function() {
    if(!this._overrayTilemap) return;
    this._overrayTileset = $gameMap.overrayTileset();
    if (this._overrayTileset) {
      var tilesetNames = this._overrayTileset.tilesetNames;
      for (var i = 0; i < tilesetNames.length; i++) {
        this._overrayTilemap.bitmaps[i] = ImageManager.loadTileset(tilesetNames[i]);
      }
      var newTilesetFlags = $gameMap.overrayTilesetFlags();
      this._overrayTilemap.refreshTileset();
      if (!this._overrayTilemap.flags.equals(newTilesetFlags)) {
        this._overrayTilemap.refresh();
      }
      this._overrayTilemap.flags = newTilesetFlags;
    }
  };

  var _Spriteset_Map_updateTilemap = Spriteset_Map.prototype.updateTilemap;
  Spriteset_Map.prototype.updateTilemap = function() {
    _Spriteset_Map_updateTilemap.call(this);
    this.checkAndCreateOverrayTilemap();
    if(!this._overrayTilemap) return;
    this._overrayTilemap.origin.x = $gameMap.overrayX() * $gameMap.tileWidth();
    this._overrayTilemap.origin.y = $gameMap.overrayY() * $gameMap.tileHeight();
    this._overrayTilemap.visible = $gameMap.isOverrayVisible();
  };

  var _Spriteset_Map_updateTileset = Spriteset_Map.prototype.updateTileset;
  Spriteset_Map.prototype.updateTileset = function() {
    _Spriteset_Map_updateTileset.call(this);
    if(!this._overrayTilemap) return;
    if (this._overrayTileset !== $gameMap.overrayTileset()) {
      this.loadOverrayTileset();
    }
  };

//////////////////////////////////////////////////////////////////////////////////////
// 点数表示とか（別プラグインにしてもいいけど面倒なので）
//////////////////////////////////////////////////////////////////////////////////////
  // フレーム更新
  var _showViewers_Game_Map_update = Game_Map.prototype.update;
  Game_Map.prototype.update = function(sceneActive) {
    _showViewers_Game_Map_update.call(this, sceneActive);
    this.updateShowViewers();
  };

  Game_Map.prototype.updateShowViewers = function() {
    var battler =  $gamePlayer.battler();
    if(TMPlugin.Shooting.SmoothMode && battler && battler.isAlive()){
      // 閲覧数がじわじわ変化
      $gamePlayer.pinchLevel();
      var viewer_plus = (Math.random() * 100) < ($gamePlayer.pinchLevel() + 1) ** 2;
      if (viewer_plus) $gameParty.gainGold(1);
    }
    if ($gameScreen.picture(TMPlugin.Shooting.ViewersPictNum) && /^b\d$/.test($gameScreen.picture(TMPlugin.Shooting.ViewersPictNum).name()) ){
      // 閲覧数表示
      var viewers = $gameParty.gold();
      viewers = viewers.clamp(0,999999); // 表示カンストは999999
      for (var i=TMPlugin.Shooting.ViewersPictNum;i<TMPlugin.Shooting.ViewersPictNum+6;i++){
        $gameScreen.picture(i)._name = 'b'+ (Math.floor(viewers%10));
        $gameScreen.picture(i)._opacity = (viewers == 0 && i != TMPlugin.Shooting.ViewersPictNum) ? 0 : 255;
        viewers = Math.floor(viewers / 10);
      }
    }
  };

//////////////////////////////////////////////////////////////////////////////////////
// 女王シャウトのシステムを用意するよ
//////////////////////////////////////////////////////////////////////////////////////

  //-----------------------------------------------------------------------------
  // Game_Map
  //

var _guang_QueenShout_Game_Map_initialize = Game_Map.prototype.initialize;
Game_Map.prototype.initialize = function() {
    _guang_QueenShout_Game_Map_initialize.call(this);
    this._queenShoutStack = [];
    this._queenShoutInterpreter =  new Game_Interpreter();
    this._queenShoutData = {};
    this.resetQueenShoutData();
    this._normalShoutInterval = 0;
    this._attackShoutInterval = 0;
    this._beatShoutInterval = TMPlugin.Shooting.QueenShoutBeatInterval;
};

Game_Map.prototype.resetQueenShoutData = function() {
  // LV0 被弾のみレベルの数値を1加算しておく事にする（ダメージを受けた直後に呼ばれるので）
    this._queenShoutData[[0,"開始"]] = {priority:8 ,bigClass: 21, smallClassTimes: [ 1, 1, 1, 1, 1] };
    this._queenShoutData[[0,"攻撃"]] = {priority:1 ,bigClass: 22, smallClassTimes: [-1,-1] };
    this._queenShoutData[[1,"被弾"]] = {priority:9 ,bigClass: 23, smallClassTimes: [-1,-1,-1] };
    this._queenShoutData[[0,"初回"]] = {priority:8 ,bigClass: 24, smallClassTimes: [ 1] };
    this._queenShoutData[[0,"撃破"]] = {priority:1 ,bigClass: 25, smallClassTimes: [-1,-1] };
    this._queenShoutData[[0,"ミサイル"]] = {priority:3 ,bigClass: 40, smallClassTimes: [-1,-1] };
    this._queenShoutData[[0,"vs姫"]] = {priority:9 ,bigClass: 41, smallClassTimes: [-1] };
    this._queenShoutData[[0,"入手"]] = {priority:2 ,bigClass: 42, smallClassTimes: [-1,-1] };
  // LV1
    this._queenShoutData[[1,"攻撃"]] = {priority:1 ,bigClass: 26, smallClassTimes: [-1,-1] };
    this._queenShoutData[[2,"被弾"]] = {priority:9 ,bigClass: 27, smallClassTimes: [-1,-1,-1] };
    this._queenShoutData[[1,"初回"]] = {priority:8 ,bigClass: 28, smallClassTimes: [-1,-1,-1] };
    this._queenShoutData[[1,"撃破"]] = {priority:1 ,bigClass: 29, smallClassTimes: [-1] };
    this._queenShoutData[[1,"ミサイル"]] = {priority:3 ,bigClass: 40, smallClassTimes: [-1,-1] };
    this._queenShoutData[[1,"vs姫"]] = {priority:9 ,bigClass: 41, smallClassTimes: [-1] };
    this._queenShoutData[[1,"入手"]] = {priority:2 ,bigClass: 42, smallClassTimes: [-1,-1] };
  // LV2
    this._queenShoutData[[2,"攻撃"]] = {priority:1 ,bigClass: 30, smallClassTimes: [-1,-1,-1,-1] };
    this._queenShoutData[[3,"被弾"]] = {priority:9 ,bigClass: 31, smallClassTimes: [-1,-1,-1] };
    this._queenShoutData[[2,"常時"]] = {priority:1 ,bigClass: 32, smallClassTimes: [-1,-1,-1] };
    this._queenShoutData[[2,"初回"]] = {priority:8 ,bigClass: 33, smallClassTimes: [-1,-1] };
    this._queenShoutData[[2,"撃破"]] = {priority:1 ,bigClass: 34, smallClassTimes: [-1] };
    this._queenShoutData[[2,"ミサイル"]] = {priority:3 ,bigClass: 40, smallClassTimes: [-1,-1] };
    this._queenShoutData[[2,"vs姫"]] = {priority:9 ,bigClass: 41, smallClassTimes: [-1] };
  // LV3
    this._queenShoutData[[3,"攻撃"]] = {priority:1 ,bigClass: 35, smallClassTimes: [-1,-1,-1,-1] };
    this._queenShoutData[[4,"被弾"]] = {priority:9 ,bigClass: 36, smallClassTimes: [-1,-1,-1] };
    this._queenShoutData[[3,"常時"]] = {priority:1 ,bigClass: 37, smallClassTimes: [-1,-1,-1] };
    this._queenShoutData[[3,"初回"]] = {priority:8 ,bigClass: 38, smallClassTimes: [-1,-1] };
    this._queenShoutData[[3,"撃破"]] = {priority:1 ,bigClass: 39, smallClassTimes: [-1] };
}

// シチュエーションからシャウトデータの分類番号を返し、そのシャウトの残り回数を減らす
Game_Map.prototype.pickUpQueensShoutClass = function(situation){
    if(!$gamePlayer.battler()) return null;
    var shout = this._queenShoutData[[$gamePlayer.pinchLevel(),situation]];
    var ret = null;
    if (shout){
        var sct = shout.smallClassTimes;
        var lottery = [];
        var max = -99999;
        for(var i = 0; i < sct.length; i++){
            if (sct[i] != 0 && max < sct[i]) max = sct[i];
        }
        for(var i = 0; i < sct.length; i++){
            if (sct[i] == max) lottery.push(i);
        }
        if (lottery.length > 0){
            var smallClass = lottery[Math.floor(Math.random() * lottery.length)];
            sct[smallClass] -= 1; // 残り回数減少
            smallClass += 1; // 実際のデータでは小分類は1からスタートしているため
            ret = [shout.bigClass,smallClass,shout.priority];
        }
    }
    return ret;
};


// シャウトの新規予約可否設定
Game_Map.prototype.enableShout = function(b){
    this._disableShout = !b;
};

// シャウトを予約追加する
Game_Map.prototype.reserveShout = function(situation){
    if(this._disableShout) return;
    var shoutClass = this.pickUpQueensShoutClass(situation);
    if (shoutClass) this._queenShoutStack.push(shoutClass);
//    console.log(situation,shoutClass);
};

// 今叫んでいるものが終わったらシャウトを終了させる
Game_Map.prototype.stopShout = function(){
    this._queenShoutStack = [];
};

// シャウトを強制終了させる
Game_Map.prototype.terminateShout = function(){
    this._queenShoutInterpreter =  new Game_Interpreter();
    this.stopShout();
    var pict = $gameScreen.picture(TMPlugin.Shooting.ShoutPictNum);
    if (pict) pict.erase();
};

// シャウト割り込み（該当シャウトが存在すれば強制終了し予約追加する）
Game_Map.prototype.interruptShout = function(situation){
    if(this._disableShout) return;
    // 優先度チェック
    var p = this._queenShoutData[[$gamePlayer.pinchLevel(),situation]];
    if(!p) return;
    p = p.priority;
    var max = (this._queenShoutInterpreter.isRunning() && this._priorityOfRunningShout) ? this._priorityOfRunningShout : 0;
    this._queenShoutStack.forEach(function(shout){
      if(max < shout[2]) max = shout[2];
    });
    if(p < max) return; // 優先度が現在再生中及びスタック中の全シャウト以上ならば割り込みを実行
    // 割り込み処理
    var shoutClass = this.pickUpQueensShoutClass(situation);
    if (shoutClass){
      this.terminateShout();
      this._queenShoutStack.push(shoutClass);
    }
};

// 常時シャウトの自動予約
Game_Map.prototype.reserveNormalShoutAuto = function(){
    if (this._queenShoutInterpreter.isRunning()) return; // スタックの余計な増加を防ぐためシャウト中は止める
    if(this._normalShoutInterval <= 0 && this._queenShoutStack.length == 0){
        this.reserveShout("常時");
        this._normalShoutInterval = TMPlugin.Shooting.QueenShoutNormalInterval * (0.9 + Math.random() * 0.2);
    }
    this._normalShoutInterval -= 1;
};

// 攻撃時シャウトの自動予約
Game_Map.prototype.reserveAttackShoutAuto = function(){
    if (this._queenShoutInterpreter.isRunning()) return; // スタックの余計な増加を防ぐためシャウト中は止める
    if(this._attackShoutInterval <= 0 && this._queenShoutStack.length == 0){
        this.reserveShout("攻撃");
        this._attackShoutInterval = TMPlugin.Shooting.QueenShoutAttackInterval * (0.9 + Math.random() * 0.2);
    }
    this._attackShoutInterval -= 1;
};

// すやぴミサイルシャウトの予約
Game_Map.prototype.reserveMissileShout = function(){
    this.interruptShout("ミサイル");
};

// 取得セリフの予約
Game_Map.prototype.reserveGetShout = function(){
    this.interruptShout("入手");

};


// 撃破時シャウトの自動予約
Game_Map.prototype.reserveBeatShoutAuto = function(){
    if (this._queenShoutInterpreter.isRunning()) return; // スタックの余計な増加を防ぐためシャウト中は止める
    if(this._beatShoutInterval <= 0 && this._queenShoutStack.length == 0){
        this.reserveShout("撃破");
        this._beatShoutInterval = TMPlugin.Shooting.QueenShoutBeatInterval * (0.9 + Math.random() * 0.2);
    }
    this._beatShoutInterval -= 1;
};

Game_Map.prototype.updateQueenShoutEvent = function() {
    if (!this._queenShoutInterpreter) return;
    if (!TMPlugin.Shooting.SmoothMode) return;

    this.reserveNormalShoutAuto();

    if (!this._queenShoutInterpreter.isRunning()) {
        if (0 < this._queenShoutStack.length){
            var shoutData = this._queenShoutStack.shift();
            var event = $dataCommonEvents[shoutData[0]];
            if (event) {
                this._queenShoutInterpreter.setup(event.list);
                $gameVariables.setValue(TMPlugin.Shooting.QueenShoutEventIdVariable, shoutData[1]);
                this._priorityOfRunningShout = shoutData[2];
            }
        }
    }else{
        this._queenShoutInterpreter.update();
    }
};

var _guang_QueenShout_Game_Map_updateInterpreter = Game_Map.prototype.updateInterpreter;
Game_Map.prototype.updateInterpreter = function() {
    _guang_QueenShout_Game_Map_updateInterpreter.call(this);
    this.updateQueenShoutEvent();
};

  //-----------------------------------------------------------------------------
  // Game_Interpreter
  //

  var _guang_QueenShout_Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function(command, args) {
    _guang_QueenShout_Game_Interpreter_pluginCommand.call(this, command, args);
    if (command === 'pushShout') {// シャウトをpush
      $gameMap.reserveShout(args[0]);
    }else if (command === 'stopShout') {// シャウトを今叫んでいるものが終わったら終了させる
      $gameMap.stopShout();
    }else if (command === 'terminateShout') {// シャウトを強制停止
      $gameMap.terminateShout();
    }else if (command === 'interruptShout') {// シャウト割り込み
      $gameMap.interruptShout(args[0]);
    }else if (command === 'resetShoutTimes') {// シャウト発動回数をリセット
      $gameMap.resetQueenShoutData();
    }else if (command === 'disableShout') { // シャウトを無効化（追加不能状態にしつつstop）
      $gameMap.enableShout(false);
      $gameMap.stopShout();
    }else if (command === 'enableShout') { // シャウトを有効化（追加可能状態に）
      $gameMap.enableShout(true);
    }
  };

//////////////////////////////////////////////////////////////////////////////////////
// シューティングモードでの早送りを無効に
//////////////////////////////////////////////////////////////////////////////////////

  //-----------------------------------------------------------------------------
  // Scene_Map
  //

var _guang_Kill_Fastforward_Scene_Map_IsFastForward = Scene_Map.prototype.isFastForward;
Scene_Map.prototype.isFastForward = function() {
    if (TMPlugin.Shooting.SmoothMode) return false;
    return _guang_Kill_Fastforward_Scene_Map_IsFastForward.call(this);
};

//////////////////////////////////////////////////////////////////////////////////////
// パワーアップアイテム
//////////////////////////////////////////////////////////////////////////////////////


  //-----------------------------------------------------------------------------
  // Game_Event
  //

var _guang_powerItem_Game_Event_UpdatePattern = Game_Event.prototype.updatePattern;
Game_Event.prototype.updatePattern = function() {
    _guang_powerItem_Game_Event_UpdatePattern.call(this);
    if (this._pattern == this.maxPattern() - 1){
      this.powerItemRoulette();
    }
};

Game_Event.prototype.powerItemRoulette = function() {
  if(!this.event().meta.item) return;
  if(this.event().meta.item == '2') return;
  var actor = $gamePlayer.battler();
  var powerUp = [];
  if($gamePlayer.hasBarrier() < 2) powerUp.push(0); // バリア
  var memids = $gameParty.members().map(function(member){return member.actorId();});
  if(!memids.includes(6)) powerUp.push(1); // ドロップ弾
  if(!memids.includes(7)) powerUp.push(2); // 2Way追加
  if( !(memids.includes(2)&&memids.includes(3)&&memids.includes(4)&&memids.includes(5)) )powerUp.push(3); // オプション
//  if( actor.weapons()[0].id != 1 ) powerUp.push(4); // ノーマル
  if( actor.weapons()[0].id != 2 && $gamePlayer.pinchLevel() >= 2 ) powerUp.push(5); // チェイサー
  if (powerUp.length == 0) powerUp = [0]; // バリア

  if(this._powerUpIndex === void 0 ) this._powerUpIndex = 0;
  this._powerUpIndex += 1;
  this._powerUpIndex %= powerUp.length;
  this._powerUpItem = powerUp[this._powerUpIndex];
  this._characterIndex = Math.floor(this._powerUpItem / 4);
  this._direction = ((this._powerUpItem % 4) + 1) * 2;
};

Game_Event.prototype.powerUpPlayer = function() {
  var actor = $gamePlayer.battler();
  if (this._powerUpItem === void 0) this.powerItemRoulette();
  if (this._powerUpItem == 0) $gamePlayer.setBarrier(2);
  if (this._powerUpItem == 1) $gameParty.addActor(6);
  if (this._powerUpItem == 2) $gameParty.addActor(7);
  if (this._powerUpItem == 3) {
    var memids = $gameParty.members().map(function(member){return member.actorId();});
    if(!memids.includes(2)){
      $gameParty.addActor(2);
    }else if(!memids.includes(3)){
      $gameParty.addActor(3);
    }else if(!memids.includes(4)){
      $gameParty.addActor(4);
    }else if(!memids.includes(5)){
      $gameParty.addActor(5);
    }
    $gameParty.members().forEach(function(mem){
      if(!mem.actor().meta.smallSuyapiId) return;
      mem.forceChangeEquip(0, $dataWeapons[actor.weapons()[0].id]);
    });
  }
  if (this._powerUpItem == 4) { // ノーマル
    $gameParty.members().forEach(function(mem){
      if(!mem.actor().meta.smallSuyapiId && mem !== $gamePlayer.battler()) return;
      mem.forceChangeEquip(0, $dataWeapons[1]);
    });
  }
  if (this._powerUpItem == 5) { // チェイサー
    $gameParty.members().forEach(function(mem){
      if(!mem.actor().meta.smallSuyapiId && mem !== $gamePlayer.battler()) return;
      mem.forceChangeEquip(0, $dataWeapons[2]);
    });
  }
  $gamePlayer.requestAnimation([7,8,4,6,0,5][this._powerUpItem]);
};


//////////////////////////////////////////////////////////////////////////////////////
// MPゲージ
//////////////////////////////////////////////////////////////////////////////////////

  //-----------------------------------------------------------------------------
  // Spriteset_Map
  //

  var _MP_Gauge_Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
  Spriteset_Map.prototype.createLowerLayer = function() {
    _MP_Gauge_Spriteset_Map_createLowerLayer.call(this);
    this.create_MpGauge();
  };

  // MPゲージの作成
  Spriteset_Map.prototype.create_MpGauge = function() {
    this._suyapiGauge = new Sprite_Suyapi_Gauge();
    this.addChild(this._suyapiGauge); // LowerLayerの最後、Pictureの前にaddしてみる
  };

  //-----------------------------------------------------------------------------
  // Sprite_Suyapi_Gauge
  //

  function Sprite_Suyapi_Gauge() {
    this.initialize.apply(this, arguments);
  };

  Sprite_Suyapi_Gauge.prototype = Object.create(Sprite_Base.prototype);
  Sprite_Suyapi_Gauge.prototype.constructor = Sprite_Suyapi_Gauge;

  Sprite_Suyapi_Gauge.prototype.initialize = function() {
    Sprite_Base.prototype.initialize.call(this);
    this.bitmap = ImageManager.loadSystem("suyapiGauge");
    this.setFrame(0, this.bitmap.height / 2, this.bitmap.width, this.bitmap.height / 2);
    this._suyapi = new Sprite();
    this._suyapi.bitmap = ImageManager.loadSystem("suyapiGauge");
    this._suyapi.setFrame(0, 0, this._suyapi.bitmap.width, 4);
    this._suyapi.y = this.height;
    this._suyapi.anchor.y = 1;
    this.addChild(this._suyapi);
    this.x = 184;
    this.y = 18;
    this._framecount = 0;
    this.visible = false;
    this._suyapi.visible = this.visible;
    this.update();
  };

  Sprite_Suyapi_Gauge.prototype.update = function() {
    var battler = $gamePlayer.battler();
    if (!battler) return;
    var max_gauge_height = (this._suyapi.bitmap.height / 2) - 8;
    var gauge_height = (Math.floor(max_gauge_height * (battler.mp / battler.mmp) / 2)) * 2;
    this._suyapi.setFrame(0, 4 + max_gauge_height - gauge_height, this._suyapi.bitmap.width, 4 + gauge_height);
    this.visible = TMPlugin.Shooting.SmoothMode;
    this._suyapi.visible = this.visible;
    if (battler.mp == battler.mmp) {
      this._framecount += 1;
      var a = (1+Math.sin(this._framecount/20))/2;
      this.setBlendColor([255,255,255,255 * a]);
      this._suyapi.setBlendColor([255,255,255,255 * a]);
    }else if (this._framecount != 0){
      this._framecount = 0;
      this.setBlendColor([0,0,0,0]);
      this._suyapi.setBlendColor([0,0,0,0]);
    }
  };

//////////////////////////////////////////////////////////////////////////////////////
// うおおおおおおおおおおおおおおおおおおおお
//////////////////////////////////////////////////////////////////////////////////////

Game_Action.prototype.itemEffectAddNormalState = function(target, effect) {
    var chance = effect.value1;
//    if (!this.isCertainHit()) {
        chance *= target.stateRate(effect.dataId);
        chance *= this.lukEffectRate(target);
//    }
    if (Math.random() < chance) {
        target.addState(effect.dataId);
        this.makeSuccess(target);
    }
};